#! /bin/sh -- # -*- perl -*-
# first line invokes sh.  Some systems may need to refer to perl directly, i.e.:

#! /bin/perl

# But the first way is preferable, because we have a trick (below) that
# allows perl to be _anywhere_ in the user's path.  The intent is to reduce
# the need to edit this file.

eval 'exec perl $0 ${1+"$@"}'
if 0;  # tee hee! This is on a separate line, so /bin/sh never sees it.

# use strict;

#-----------------------------------------------
# DESCRIPTION

# Tired of spam and other net abuse?  Report it!  Most service providers
# terminate accounts used for such purposes.

# Adcomplain is a tool for reporting inappropriate commercial e-mail and
# Usenet postings, as well as chain letters and "make money fast" postings.

# It automatically analyzes the message, composes an abuse report, and mails
# the report to the offender's internet service provider.  The report is
# displayed for your approval prior to mailing.  Adcomplain can be invoked
# from the command line or automatically from many news and mail readers.

# Adcomplain runs under Unix, Windows-NT, Windows-95, and OS/2.  Adcomplain
# does _not_ currently run on Macintosh systems.  [URL for a Macintosh solution
# TBA...]

# You must have Perl  installed on your system to run this.  (Either perl
# version, 4 or 5, will work.)  Perl can be obtained at:  http://www.perl.com/

# adcomplain was created in the belief that a single concise message is the
# most appropriate way to complain.  Mail bombing (e-mailing megabytes of
# useless data) and public flaming (replying on Usenet, causing your complaint
# to be duplicated on every machine in the network) are discouraged.

# The default letter meets ISP standards (such as those required by netcom) for
# abuse reports.  The letter hopefully makes its point without wasting value
# computer or human bandwidth.  It marks itself as being machine-generated, so
# that system administrators can quickly recognize its format and content.

# A third-party forwarding service called "abuse.net" is used for complaints to
# the offender's provider.  This ensures that the best known complaint address
# is used.  The first time you use abuse.net, you will receive a message asking
# you to register.  See http://www.abuse.net for more information.

#-----------------------------------------------
# USAGE
# Pipe the Usenet posting or e-mail into this program.  It may detect
# mail forgeries or other problems and ask you to confirm by pressing return.
# It will then compose the message and display it for approval before mailing.
# You have opportunities to edit or list the message, or to just abort.

# SYNOPSIS
# adcomplain [-b] [-c] [-f template] [-m] [-o outname] [-p inname] [-q] [-s]
#	[-u] [-v] [newsgroup]

# OPTIONS
# Any combination of options may be used together. For example, -c and
# -m may be used together to complain about a chain letter received via
# e-mail.

# You may supply a newsgroup name (or comma separated newsgroups list)
# as the last argument. If not supplied, the contents of the
# 'Newsgroups' header, if present, will be used. 

# -b
#   Batch mode.  This is not recommended; adcomplain is not as smart as
#   a human, and automated responses will often be sent to an inappropriate
#   site.  This switch has been added by user demand, but your are implored
#   in the strongest possible terms to originate, examine, and approve each
#   one of your complaints individually.  Violations of terms of service are
#   serious offenses, and each complaint deserves your personal attention.

# -c
#   Chain letter mode. Composes and mails a complaint about chain
#   letters, such as Make Money Fast.

# -f template
#   Use the contents of the specified file as the complaint text.
#   The contents of this file will have perl variables expanded before
#   proceeding.  Some useful ones are:
#
#   $header_analysis -- list of header anomalies in the message
#   $newsgroups -- news groups this was posted to
#   $postmaster -- postmaster at the originator's site

# -l
#   (Legal letter)  Set this if the e-mail or posting is legal, but
#   merely inappropriate.  If this is not set, mail will be cc'd to
#   the authorities (see $authorities).

# -m
#   Force the input to be regarded as a mail message.  This is not
#   necessary in most instances; adcomplain will correctly infer whether the
#   message is a mail or a Usenet message.

# -o outname
# 	Specify a file in which to place the generated complaint message;
#	this option also prevents adcomplain from mailing the complaint.

# -p inname
#  Read the ad from given file.  You _must_ use this option on
#  Windows-NT, Windows-95, and OS/2 systems.

# -q
#   Quiet mode.  Doesn't ask any questions except whether to send the message.
#   You must still examine the message, and you are given an opportunity to
#   modify the message or abort it.

# -s
#   Don't append your .signature.  Adcomplain normally appends your signature
#   (though this behavior can be changed at configuration time);  this switch
#   ensures that your signature is omitted.

# -u
#   Don't resolve.  Regard the message syntactically, i.e. don't try to confirm
#   host names and domains.  This is useful if you are behind a firewall
#   and have absolutely no way to confirm the hostnames within the message.

# -v
#   Verbose mode. Diagnostic messages will be printed.

# NOTE: the -t switch is a regression test mode.  It runs the header analysis
# in verbose mode and stops before sending the message.  It is not intended
# for normal use.

#-----------------------------------------------
# ENVIRONMENT VARIABLES
# The following environment variables influence the operation:

# $ADCOMPLAIN_DOMAINS is used to set a list of domain names that should _not_
# be construed as valid origination domains for spam.  The default is
# empty.  The list should be space-delimited, i.e.,
#
# (in Windows-NT, Windows-95, and OS/2):
# set ADCOMPLAIN_DOMAINS=domain1.com domain2.com
# (Unix, ksh):
# export ADCOMPLAIN_DOMAINS="domain1.com domain2.com"

# $ADCOMPLAIN_FROM will specify the "From:" address in the resulting complaint
# message.  This is a convenience for individuals who use adcomplain in
# multiple mail domains but want to register with abuse.net only once.
# Users on Windows-95 _must_ set this.

# $ADCOMPLAIN_HOSTS is a file that is searched before the regular
# gethostbyname() # and gethostbyaddr() functions are called.  (This is
# mainly used by adcomplain's internal regression test suite.)

# $ADCOMPLAIN_MAILHOST is the name of the host to use to relay (SMTP)
# our complaints.  The default is the current host or the value of $mailhost
# in the configuration section below.
# Win-95 and Win-NT users must almost always set this, because these systems
# do not have a sendmail daemon.

# $ADCOMPLAIN_MDOMAIN will specify your mail domain.  The intended use
# for this would be in a shared installation where multiple hosts use one
# copy of this script, but adcomplain is unable to determine your mail domain
# on its own.  This is not needed if $ADCOMPLAIN_FROM is set.  Administrators
# may wish to set this parameter directly in the script, i.e.:
#  $ENV{'ADCOMPLAIN_MDOMAIN'} = 'juno.com';

# $ADCOMPLAIN_OPTS is an additional list of switches that should be added to
# every invocation of adcomplain.

# $EDITOR -- the editor to use to edit the message, if $VISUAL is not
# defined.  If not defined, the default is "vi" on Unix, "notepad" on
# win32, and "e" on OS/2.

# $PAGER -- The message is always displayed prior to mailing using this
# program.  If $PAGER is not defined, the default is "more" on Unix,
# and the internal pager on Windows-NT, Windows-95, and OS/2.  You can specify
# the builtin pager by setting $PAGER to "builtin".

# $SIGNATURE -- name of a file that holds your signature.  The default
# is $HOME/.signature.

# $VISUAL -- the editor to use to edit the message, when an editor is requested.
# If not defined, $EDITOR is used.

#-----------------------------------------------
# INSTALLATION
# You must have either perl 4.036 or perl 5.0 installed.
# Perl can be obtained at:  http://www.perl.com/

# Installation on Windows-NT, Windows-95, and OS/2 Systems:
# --------------------------------------------------
# Save this script in a convenient place on your system.

# Make sure your system is configured to run TCP/IP, and that you
# have perl5 installed or your system.

# If you are running on Windows 95, you _must_ set the following
# environment variables:
#	COMPUTERNAME -- the name of your host, as used in TCP/IP
#	ADCOMPLAIN_FROM -- your return e-mail address

# If you are running on OS/2, you _must_ set the following environment
# variables:
#	HOSTNAME -- the name of your host, as used in TCP/IP
#	ADCOMPLAIN_FROM -- your return e-mail address

# You can either put a line like
#	$ENV{'ADCOMPLAIN_FROM'} = 'myname@somedomain.com';
# in the configuration section, or put "set" commands in your autoexec file
# and reboot.  Your pick.

# Examine the configuration section carefully and adjust to taste.  Note
# that you _must_ set $mailhost, unless you've actually bought and
# installed a sendmail program on your machine.

# To test, send an e-mail message to yourself, save the message in a
# file, and type "adcomplain -p file", where "file" is the file in which
# you saved the message.  Edit the resulting message to send the complaint
# to yourself, and let adcomplain mail the message.  You should receive the
# mail message shortly.

# Installation on Unix Systems:
# -----------------------------
# Type "echo $PATH" to list the directories in your search path.
#
# Go to one of those directories and save this script in a file named
# "adcomplain".
#
# Examine the Configuration Section and make sure it is correct for your
# system.  You will probably want to change the value for $mailhost to
# match your system.
#
# Type "chmod +x adcomplain" to make it executable.
#
# Type "rehash" to add it to the search path.
#
# To test, send an e-mail message to yourself, save the message in a
# file, and type "adcomplain <file", where "file" is the file in which
# you saved the message.  Edit the resulting message to send the complaint
# to yourself, and let adcomplain mail the message.  You should receive the
# mail message shortly.

# There are reports that the complex "#!" exec-line causes trouble on some
# primitive BSD systems.  You may have to find perl (try "/bin/which perl"
# "/usr/bin/which perl", or "type perl") and modify the first line per the
# comments at the front of this file.

#-----------------------------------------------
# NEWSREADER CONFIGURATION
# You can configure many newsreaders to allow easy response to Usenet spam.

# =================
# rn, trn, and strn:

# Put the following in your ~/.rnmac file to map adcomplain to the 'a' key:
#
# a %(%m=[pa]?|adcomplain %C\n:a)
#
# In addition, rn users can complain about chain letters by mapping the
# 'A' key as follows:
#
# A %(%m=[pa]?|adcomplain -c\n:A)
#
# rn users can also customize the message based on the current newsgroup
#   by using adcomplain's -f mode as shown in the following example:
#
# Z %(%m=[pa]?%(%C=aaa.forsale?|adcomplain -f aaa_complaint\n:%(%C=bbb.forsale?|adcomplain -f bbb_complaint\n:%(%C=ccc.forsale?|adcomplain -f ccc_complaint\n:%(%C=ddd.forsale?|adcomplain -f ddd_complaint\n:Z)))))
#
# When 'Z' is pressed, the above macro checks the newsgroup name, and if
# it matches one of four choices (aaa.forsale, bbb.forsale, ccc.forsale,
# ddd.forsale), uses a complaint message specific to that newsgroup.
# You can use it to complain about foreign postings in local newsgroups,
# for example (e.g., "aaa.forsale is for items for sale in the Podunk
# area.")

# =================
# nn:

# Put the following in your ~/.nn/init file to map adcomplain
# to ctrl-A and chain mail complain to ctrl-B:
# 
# map both ^A (
# :!! cat /dev/null > /tmp/junk$(USER)
# save-full "//tmp/junk$(USER)"
# :! echo; adcomplain $G < /tmp/junk$(USER)
# :!! rm -rf /tmp/junk$(USER)
# )
# map both ^B (
# :!! cat /dev/null > /tmp/junk$(USER)
# save-full "//tmp/junk$(USER)"
# :! echo; adcomplain -c $G < /tmp/junk$(USER)
# :!! rm -rf /tmp/junk$(USER)
# )
#

# =================
# tin:

# Type "|" at the article level to pipe the article into
# adcomplain. Type "adcomplain" when it asks for the command name.

# =================
# SLRN newsreader (thanks to Ralf Hildebrandt <R.Hildebrandt@tu-bs.de>)

# Put an additional line in the ~/.slrnrc :
#	interpret ".slrn.sl"
#
# The file ~/.slrn.sl contains:
#
# define pipe_to_adcomplain ()
# {
#    pipe_article ("adcomplain -q");
# }
# definekey ("pipe_to_adcomplain", "S", "article");
#
# This binds the key "S" to adcomplain -q "article".

#-----------------------------------------------
# DISCLAIMER, LICENSE, AND TERMS OF USE

# The latest version of the Gnu Copyright ("copyleft") hereby constitutes
# your terms of use and redistribution.  The Gnu Copyright, which is included
# with each Free Software Foundation distribution, is available from your
# favorite Gnu archive.

# IN NON-LEGAL TERMS, the Gnu Copyright gives you the right to use,
# redistribute, modify, or even borrow part or all of this program.  You
# are explicitly prohibited from selling this program or incorporating parts
# of it in software for profit.  Read the copyright for the exact terms and
# conditions.

# When you use this program you assume all risk and liability; it has no
# warranty.  A portion of the Gnu Copyright addressing the warranty is herein
# reproduced:

#                             NO WARRANTY
# 
#   11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
# FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
# OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
# PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
# OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
# TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
# PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
# REPAIR OR CORRECTION.
# 
#   12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
# WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
# REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
# INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
# TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
# YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
# PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGES.

#-----------------------------------------------
# BACKGROUND, CREDITS, and BUG REPORTING

# adcomplain is the brainchild of "Bill McFadden" billmc@agora.rdrop.com,
# who wrote it originally in csh (!!!).

# "D. Jason Penney" <jason@gemstone.com> transcribed it to run in perl,
# significantly augmented the header processing, and added domain name
# verification.

# Acknowledgements and thanks also go out to:
#   "Denis N. Antonioli" <antonioli@ifi.unizh.ch>
#   "Andrey A. Chernov" <ache@nietzsche.net>
#   "Jeff Salisbury" <jeff@cobra.sdl.usu.edu>
#   "Kelly Roach" <kbroach@daisy.uwaterloo.ca>
#   "Martin Ward" <Martin.Ward@durham.ac.uk>
#   "Wolfgang Weisselberg" <weissel@ph-cip.Uni-Koeln.DE>
#   "Wm" <wm_tarr@bigfoot.com>

# The latest version of this file is available at:
# http://www.rdrop.com/users/billmc

# Please mail ideas and bug reports to
# billmc@agora.rdrop.com or jason@gemstone.com.

# Be sure to include the version number and a test case in your bug report.

# $Id: adcomplain.pl,v 3.52 1999/05/10 16:51:28 jason Exp $
#-----------------------------------------------
local(
  $addsig, 		# set if you want ~/.signature to be added
  $cc, 			# list of cc'd recipiets
  $complaintFile, 	# predefined complaint text, if you don't like ours
  $fqdnWithoutHost,	# whether the fqdn should include the name of the host
  $mailhost, 		# host that is running your sendmail daemon
  $mailscript, 		# program for delivering the mail
  @mydomains, 		# list of domains that map to your current domain
  $noinclude, 		# set if you don't want to copy the ad in the letter
  $recipient		# whether it should go to poster, postmaster, or both
  );

#-----------------------------------------------
# CONFIGURATION SECTION
#-----------------------------------------------

# This section allows you to customize the default operation at your site.
# Example lines are included in comments, and a normal set of defaults are
# currently active.  Please examine each default and make a decision if
# the current settings are appropriate to your system.

# Default host for smtp traffic?  This is specific to GemStone; you
# will probably want to delete this.  This is overridden by
# $ADCOMPLAIN_MAILHOST.
#$mailhost = "mailhost";

# Include your .signature file?
# Set to 0 if you don't want to include your signature.
# Normally, adcomplain will append your signature unless -s is specified.
# $addsig = 0;
$addsig = 1;

# Are there authorities to which you should report illegal e-mails such
# as pyramid schemes and unsolicited commercial e-mail?  Comma-separated
# if more than one.
$authorities = "uce\@ftc.gov"; # Consult http://www.ftc.gov for details.
# $authorities = ""; # ...ftc.gov is only appropriate if your are in the USA.

# CC List -- Do you want a copy of the complaint sent to someone else?
# This is comma-separated if there is more than one.
$cc = "";

# Customize your complaint text?
# This is the default name of the file to use for the complaint text.  If the
# file name is empty, the default text will be used.  Override with the
# "-f <filename>" switch.
$complaintFile = "";

# Customize mail delivery?
# Adcomplain normally contacts a mail delivery agent directly.  If $mailscript
# is set, adcomplain runs the specified program directly instead, once per
# recipient.  Two arguments are passed:
#   1. The name of a file to send.  It contains all of the sendmail headers,
#      so your program will have to use or modify them appropriately.
#   2. "batch" or "interactive", in case the script wants to make
#      different UI decisions
# Suggested usage: PGP-signing the mail message, ...?
# $mailscript = "/usr/lib/sendmail -t <";

# Friendly Mail Domains?
# One spammer trick is to directly originate the mail message on your local
# host.  In other words, they contact the MTA at your mail site directly and
# originate the message at your site.  This is similar to the way adcomplain
# sends your response ;-).  If your MTA does not have appropriate security
# measures, adcomplain might decide that your complaint should be addressed to
# your own postmaster!  This is a list of domain names for your system; names
# in this list are excluded from consideration when selecting a complaint
# address.
#
# It may also be desirable to set this list if your domain has multiple names,
# or if you regularly get mail forwarded from another domain.
#
# If the ADCOMPLAIN_DOMAINS environment variable is set, it will be read
# as a space-delimited list of domains to use and will be _added_ to the
# default list set here.
#
# an example:  @mydomains = ('gemstone.com', 'slc.com');
@mydomains = ();

# Always include a copy of the offending message?
# Set to 0 to include a copy of the posting every time.
# Ordinarily this is not done unless you choose to notify the provider.
# $noinclude = 0;
$noinclude = 1;

# Default Recipient(s)
# Do you want to complain to the poster, to his mail domain, or both?
# $recipient = 1; # ($complain_to_only) only to the authorities;
# $recipient = 2; # ($poster_only) only to the poster
# $recipient = 3; # ($poster_and_complain_to) both 
$recipient = 1; # ($complain_to_only) only to the authorities;

# Degree of qualification of the hostname
# Some domains require the fqdn to include the name of the
# computer to correctly deliver the mail. Others don't.
$fqdnWithoutHost = 0;
#$fqdnWithoutHost = 1;

#-----------------------------------------------
# NO USER SERVICEABLE PARTS INSIDE
#-----------------------------------------------
local(
  $adcomplain_name,     # revision number of this adcomplain
  $adcomplain_revision, # revision number of this adcomplain
  $EXEMPT_LIST, 	# list of domains we don't send complaints to
  %available_hosts, 	# list of possible sites to complain to
  $batchmode, 		# control flag, indicates no interactive input
  $bullet, 		# constant, used when formatting letter
  $bullet2,		# constant, used when formatting letter
  $chainmode, 		# control flag, indicates this is a chain letter
  $complain_to, 	# name we chose to send the complaint to
  $complain_to_only, 	# constant value for $recipient: no complaints to poster
  $dont_resolve, 	# control flag, indicates don't try gethostbyname() etc.
  $editor, 		# name of your editor, for editing the letter
  $from, 		# your e-mail address
  $header_analysis, 	# list of non-RFC compliant problems with the ad
  $hosts_file, 		# list of hosts to be tried before gethostbyname()
  %hosts_to_ips, 	# cached list of results from gethostbyname()
  %ips_to_hosts, 	# cached list of results from gethostbyaddr()
  $local_hostname,	# name of this host (best guess)
  $mailmode,		# optional control flag, indicates this is not USEnet
  @monthNames, 		# constant list of months, as per RFC 822
  $newsgroup,		# name of the newsgroup(s) this ad was posted to
  $omit_signature, 	# control flag, ensures ~/.signature not in your letter
  $opt_err,		# getopt; indicates error parsing options
  $opt_b, 		# getopt (-b, sets $batchmode)
  $opt_c, 		# getopt (-c, sets $chainmode)
  $opt_f, 		# getopt (-f, sets $complaintFile)
  $opt_h, 		# getopt (-h, prints help and exits)
  $opt_l,		# getopt (-l, sets $legal_mode)
  $opt_m, 		# getopt (-m, sets $mailmode)
  $opt_o, 		# getopt (-o, sets $outmode and $outfile)
  $opt_p, 		# getopt (-p)
  $opt_q, 		# getopt (-q, sets $quiet)
  $opt_s,		# getopt (-s, sets $omit_signature)
  $opt_t, 		# getopt (-t, clears $quiet, sets $verbose, $testMode)
  $opt_u, 		# getopt (-u, sets $dont_resolve)
  $opt_v, 		# getopt (-v, sets $verbose)
  $os_kind, 		# the type of OS we're running on
  $outfile, 		# file we write letter to, if $outmode set
  $outmode, 		# control flag, letter gets saved to file, not mailed
  $pager,		# name of your paging program, for displaying letter
  $poster, 		# our guess as to the name of the poster
  $poster_and_complain_to, # constant value for $recipient
  $poster_only, 	# constant value for $recipient
  $postmaster,		# name of postmaster at complaint address
  $quiet, 		# control flag, limits interaction and output
  %rematch_aliases,	# heuristic list, resets priority of another header
  $response, 		# text of letter we will send
  $site, 		# site we will send the complaint to
  @site_exceptions,	# sites that have irregular syntactic form
  $subject, 		# subject header of the ad
  $tempname, 		# name of a temp file, esp. for displaying the letter
  $testMode,		# control flag, omits the mail sending step
  %timezones, 		# constant list of RFC-822 compliant time zone names
  @valid_domains,	# the known valid domain suffixes for countries
  $verbose, 		# control flag, gives noisy narrative of analysis
  @wdayNames, 		# constant list of RFC-822 compliant week day names
  );

&initialize();		# set up our constants, lists, environment
&process_inputs(); 	# parse command line options, read the message
&parse_headers();	# Analyze the message
&check_hosts();		# Analyze the list of possible hosts
&summarize();		# brief discussion of problems found

$response = &compose_message();	# Create the text of the message.

# Place a copy of the message on the disk
open(TEXTFILE, ">$tempname") || die "cannot open $tempname: $!";
printf TEXTFILE "%s", $response;
close TEXTFILE;

# Display the message
if (!$batchmode && !$outmode) {
  print "\n";
  &moref($tempname);
  }

&deliver_message();	# Display, mail, edit, abort...

print "\n";
unlink $tempname;
exit 0;

#-----------------------------------------------
# Process command line options, read the input file.
sub process_inputs {
  local($junk, @junk, $i);

  $junk = $ENV{"ADCOMPLAIN_OPTS"};
  if (defined($junk) && $junk ne "") {
    @junk = split(/\s/, $junk);
    # push(@ARGV, @junk);
    unshift(@ARGV, @junk); # make sure that non-dash opts stay at end
    }
  &Getopts('bcf:hlmo:p:qstuv');
  if ((defined($opt_h) && $opt_h) || $opt_err) {
    print "Usage: $0 [-b] [-c] [-f template] [-o outfile] [-p infile]\n";
    print "                   [-hmqv] [newsgroup]\n";
    print " -b -- batch mode (not recommended)\n";
    print " -c -- chain letter\n";
    print " -f -- text to use instead of standard blurb\n";
    print " -h -- this message\n";
    print " -l -- legal mode, don't cc the authorities ($authorities)\n";
    print " -m -- force input to be regarded as a mail message\n";
    print " -o -- write mail message to a file and quit without mailing\n";
    print " -p -- read the ad from given file\n";
    print " -q -- ask no unnecessary questions\n";
    print " -s -- Omit .signature from resulting letter.\n";
    # -t undocumented: it is a test-mode switch: adcomplain stops after
    # composing the letter but before attempting to mail it.  It is used
    # internally for regression testing.
    print " -u -- Don't attempt to use gethostbyaddr name resolution\n";
    print " -v -- verbose\n";
    print "\n";
    print "Version: $adcomplain_revision\n";
    exit 0;
    }

  $batchmode = 0; $batchmode = 1 if defined($opt_b) && $opt_b;
  $chainmode = 0; $chainmode = 1 if defined($opt_c) && $opt_c;
  $complaintFile = "";
  $complaintFile = $opt_f if (defined($opt_f) && $opt_f ne "");

  $legal_letter = 0; $legal_letter = 1 if defined($opt_l) && $opt_l;
  $mailmode = 0; $mailmode = 1 if defined($opt_m) && $opt_m;
  if (defined($opt_o) && $opt_o ne "") {
    $outfile = $opt_o;
    $outmode = 1;
    }
  else {
    $outfile = "";
    $outmode = 0;
    }
  $quiet = 0; $quiet = 1 if defined($opt_q) && $opt_q;
  $omit_signature = 0; $omit_signature = 1 if defined($opt_s) && $opt_s;
  $dont_resolve = 0; $dont_resolve = 1 if defined($opt_u) && $opt_u;
  $verbose = 0; $verbose = 1 if defined($opt_v) && $opt_v;

  # Resolve conflicting modes
  if (defined($opt_t) && $opt_t) { # test mode
    $quiet = 0;
    $verbose = 1;	# print traces
    $testMode = 1;# quits without mailing
    }

  $junk = $ENV{"ADCOMPLAIN_DOMAINS"};
  if (defined($junk) && $junk ne "") {
    @junk = split(/\s/, $junk);
    push(@mydomains, @junk);
    for ($i = 0; $i < @mydomains; $i ++) {
      $mydomains[$i] = &find_site($mydomains[$i]);
      }
    }
  if (@mydomains != 0 && $verbose) {
    print "adcomplain: home domains: ";
    for (@mydomains) {
      print "$_ ";
      }
    print "\n";
    }

  # copy original Usenet posting from stdin
  if ($opt_p) {
    open(MSG, "<$opt_p") || die "cannot open $opt_p: $!";
    @original = <MSG>;
    close MSG;
    }
  elsif ($os_kind eq "unix") {
    @original = <STDIN>;
    if (!$batchmode) {
      open(STDIN, $CONSOLE) || die "cannot reopen tty"; # UNIX dependency...
      }
    }
  elsif ($os_kind eq "winnt" || $os_kind eq "win32" || $os_kind eq "os2") {
    die "The -p option is required." unless defined($opt_p) && $opt_p ne "";
    }
  else {
    die "Don't know how to scarf stdin and reset console on this os ($os_kind)";
    }
  }

#-----------------------------------------------
# Finish up list of complaints about the message, make sure user sees them.
sub summarize {
  local($ans);

  for (sort(keys(%hosts_to_ips))) {
    $junk = $hosts_to_ips{$_};
    next if $junk ne "";
    $header_analysis .= "${bullet}I cannot find the IP of $_.\n\n";
    }
  for (sort(keys(%ips_to_hosts))) {
    $junk = $ips_to_hosts{$_};
    next if $junk ne "";
    $header_analysis .=
	"${bullet}I cannot convert IP \"$_\" to a host name.\n\n";
    }

  if (!$batchmode) {
    if ($header_analysis ne "") {
      $poster = "";
      $junk = &fmt($header_analysis, 72);
      print "adcomplain: header analysis:\n";
      print "----------------\n";
      print "$junk";
      print "----------------\n";
      }
    if ($verbose || $header_analysis ne "") {
      print "\nSummary of Potential Addressees: (\"$site\" recommended)\n";
      for (sort(keys(%available_hosts))) {
	printf "  %25s %s\n", $_, $available_hosts{$_};
	}
      }
    }

  # Limit the options
  if ($complain_to eq "" && $poster eq "") {
    if ($batchmode) {
      print "ERROR: adcomplain did not detect a valid recipient.\n";
      exit 1;
      }
    else {
      print "ERROR: no valid recipients; edit and correct before sending.\n";
      }
    }
  elsif ($complain_to eq "") {
    $recipient = $poster_only;
    }
  elsif ($poster eq "") {
    $recipient = $complain_to_only;
    }

  if ($testMode) {
    print "TEST MODE: early exit\n";
    exit 0;
    }

  # Make sure critical messages are seen
  if ($quiet || $batchmode) {
    }
  elsif ($poster eq "" || $complain_to eq "" || $header_analysis ne "") {
    print "Press enter to continue.... ";
    $ans = <STDIN>;
    }
  }

#-----------------------------------------------
# send, abort, edit, or list the message
sub deliver_message {
  local($send_to, $cc_to, $each, $ok);

  for (;;) {
    if ($batchmode) {
      $ans = "s";
      }
    else {
      print "\n";
      $ans = &get_answer("Send, abort, edit, or list (s/a/e/l)?", "a");
      }
    if ($ans =~ /s.*/ || $ans =~ /y.*/) {
      # re-extract the addressee from the body
      local(@junk, $junk);
     
      # we only need the headers, so that's all we read. We want to be able
      # to understand continued lines as well.
      open(F, "<$tempname") || die "temp file $tempname has disappeared!";
      for (;;) {	# load the headers
	$junk = <F>;
      	if (!defined($junk) || $junk =~ /^$/) { # end of headers
          push(@junk, $last_line) if $last_line ne "";
          last;
	  }
        if ($junk =~ /^(\S+: .*)/) {   # normal header line
         push(@junk, $last_line) if $last_line ne "";
         $last_line = $junk;
         next;
	 }
       if ($junk =~ /^\s+(.*)/) { # continued header line
         chop $last_line;    # kill the newline
         $last_line .= " "; 
         $last_line .= $1;
         next;
	 }
       }	# load the headers
      close F;

      $send_to = $cc_to = "";
      foreach $each (@junk) {
	last if $each =~ /^$/;		# end of headers
	if ($each =~ /^To: (.*)/i) {
	  $send_to = $1;
	  }
	elsif ($each =~ /^Cc: (.*)/i) {
	  $cc_to = $1;
	  }
	}
      if ($send_to eq "") {
	if ($batchmode) {
	  print "adcomplain: fatal error (no addressee)\n";
	  last;
	  }
	print "no addressee! re-edit and continue\n";
	next;
	}
      if ($outmode) {
	local(@out_array) = ();
	open(OUTFILE, "<$tempname") || die "error opening $tempname: $!";
	@out_array = <OUTFILE>;
	close(OUTFILE);
	open(OUTFILE, ">$outfile") || die "error creating $outfile: $!";
	print OUTFILE @out_array;
	close(OUTFILE);
	print "written to $outfile\n";
	last;  # all done!
	}

      $ok = &send_to_list($from, "$send_to, $cc_to", $tempname);
      if (!$ok) {
	print "send to $send_to failed\n";
	if ($batchmode) {
	  unlink $tempname;
	  exit 1;		# good exit status
	  }
	next;
	}
      print "sent to: $send_to\n";
      last;
      }
    if ($ans =~ /a.*/) {
      print "aborted\n";
      last;
      }
    if ($ans =~ /e.*/) {
      system "$editor $tempname <$CONSOLE";
      next;
      }
    if ($ans =~ /l.*/) {
      &moref($tempname);
      next;
      }
    print "what?\n";
    } # for
  }

#-----------------------------------------------
# Chop newline, respecting RFC-821 end of line conventions
sub rfc822_chop {
  local($text) = @_;

  for (;;) {
    return $text if $text !~ /[\n\r]$/;
    chop($text);
    }
  }

#-----------------------------------------------
# Exchange a line of text with an SMTP daemon
sub send_line {
  local($text) = @_;
  local($response);

  print S "$text\r\n";	# RFC-821, ^M^J is end of line!
  $response = <S>;
  $response = &rfc822_chop($response);
  return $response if $response =~ /^500/;
  print "send_line: response \"$response\"\n" if $verbose;
  return "";
  }

#-----------------------------------------------
# Relay a mail message through an SMTP daemon.
# Note: this is for use with perl-4.  I _know_ perl-5 has better routines.
sub do_sendmail {
  local($from, $body, @list) = @_;
  local ($each);

  local($sockaddr) = 'S n a4 x8';	# magic spell to pack an inet_addr
  local($name, $aliases, $type, $len, $thisaddr, $thataddr);
  local($local_inetaddr, $remote_inetaddr);
  local($targetPort) = "smtp";	# or 25, world-wide standard
  local($response);
  local($local_prot) = (getprotobyname('tcp'))[2];

  if (defined($mailscript) && $mailscript ne "") {
    $name = "$mailscript $body ";
    if ($batchmode) {
      $name .= "batch";
      }
    else {
      $name .= "interactive";
      }
    print "mail command: $name\n" if $verbose;
    $response = system($name) >> 256;
    if ($response != 0) {
      print "mail command \"$name\" failed!\n";
      return 0;
      }
    return 1;
    }

  ($name, $aliases, $targetPort) = getservbyname($targetPort, 'tcp')
      unless $targetPort =~ /^\d+$/;

  # Fill out local side of connection
  ($name, $aliases, $type, $len, $thisaddr) = gethostbyname($local_hostname);

  # Fill out remote side
  if ($mailhost =~ /\d+\.\d+\.\d+\.\d+/) {
    local($a, $b, $c, $d) = split(/\./, $mailhost);
    $thataddr = pack("C4", $a, $b, $c, $d);
    }
  else {
    ($name, $aliases, $type, $len, $thataddr) = gethostbyname($mailhost);
    if (!defined($name) || $name eq "") {
      # $! doesn't have error return here.  Sigh.
      print STDERR "cannot find \"$mailhost\" (gethostbyname)\n";
      return 0;
      }
    }
  
  $local_inetaddr = pack($sockaddr, &AF_INET, 0, $thisaddr);
  $remote_inetaddr = pack($sockaddr, &AF_INET, $targetPort, $thataddr);
  
  #MAKE the socket filehandle.
  if ( ! socket( S, &AF_INET, &SOCK_STREAM, $local_prot)) {
    print STDERR "socket() failure: $!\n";
    return 0;
    }
  
  #GIVE the socket an address.
  if (! bind(S, $local_inetaddr)) {
    print STDERR "bind() failure: $!\n";
    return 0;
    }
  
  #Call up the server.
  if (! connect( S, $remote_inetaddr)) {
    print STDERR "unable to connect() to \"$mailhost\": $!\n";
    return 0;
    }
  print "do_sendmail: contacted \"$mailhost\"\n" if $verbose;
  
  #Set socket to be command buffered.
  select(S); $| =1; select(STDOUT);

  # sometimes you have to be aggressive to get the daemon to talk?
  # print S "\r\nHELO $local_hostname\r\n";
  # $response = <S>;
  # print "do_sendmail: throw-away response \"$response\"\n" if $verbose;

  # Start the dance....
  print S "HELO $local_hostname\r\n";

  # Try to get the initial acknowledgement.  Some SMTP servers are silent
  # until you speak first.  Others speak without being prompted.  The best way
  # out of this ambiguity is to just eat until we see the banner.  We
  # do _not_ check for 500 errors here.
  for (;;) {
    $response = <S>;
    $response = &rfc822_chop($response);
    print "do_sendmail: initial response \"$response\"\n" if $verbose;
    last if $response =~ /^220 /;
    last if $response =~ /^250 /;
    last if $response =~ /^SMTP/;
    }

  $response = &send_line("MAIL FROM: <$from>");
  if ($response ne "") {
    print "do_sendmail: error(MAIL FROM) \"$response\" from sendmail\n";
    close S;
    return 0;
    }

  foreach $each (@list) {
    $response = &send_line("RCPT TO: $each");
    if ($response ne "") {
      print "do_sendmail: error(RCPT TO $each) \"$response\" from sendmail\n";
      close S;
      return 0;
      }
    }

  $response = &send_line("DATA");
  if ($response ne "") {
    print "do_sendmail: error(DATA) \"$response\" from sendmail\n";
    close S;
    return 0;
    }

  if (!open(F, "<$body")) {
    print "do_sendmail: cannot open $body\n";
    close S;
    return 0;
    }
  for (;;) {
    $body = <F>;
    last if !defined($body);
    $body = &rfc822_chop($body);	# put our _own_ line terminator on it
    $body = ".$body" if $body =~ /^\..*/; # rfc821 sec. 4.5.2 transparency
    print S "$body\r\n";
    }
  close F;

  $response = &send_line(".");
  if ($response ne "") {
    print "do_sendmail: error(final) \"$response\" from sendmail\n";
    close S;
    return 0;
    }

  $response = &send_line("QUIT");
  if ($response ne "") {
    print "do_sendmail: error(QUIT) \"$response\" from sendmail\n";
    close S;
    return 0;
    }
  close S;

  return 1;
  }

#-----------------------------------------------
# Deliver text to everyone on a list
sub send_to_list {
  local($from, $send_to, $body) = @_;
  local($ok);
  local(@list) = split(/[ \t\n,]+/, $send_to);

  $ok = &do_sendmail($from, $body, @list);
  return $ok;
  }

#-----------------------------------------------
# getopts.pl - a better getopt.pl.  The one shipped with perl has bugs.

# Usage:
# do Getopts('a:bc'); # -a takes arg. -b & -c not. Sets opt_* as a
# # side effect.
# $opt_err set nonzero in case of error.

sub Getopts {
  local($optlist) = @_;
  local(@args, $_, $first, $rest, $pos);
  local(@new_argv, @unused_argv);
  local(@active_argv) = ();

  # Gather up the argv's that we should be examining
  for (;;) {
    last if !@ARGV;
    $each = shift(@ARGV);
    last if ($each eq "--"); # POSIX convention
    push(@active_argv, $each);
    }
  @unused_argv = @ARGV; # remaining args unprocessed
  @new_argv = ();

  $opt_err = 0;
  @args = split(/ */, $optlist);
  for (;;) { # process argv
    last if !@active_argv;

    if (($_ = $active_argv[0]) !~ /^-(.)(.*)/) {
      # Allow non-switch args, as in gnu getopt
      push(@new_argv, $active_argv[0]);
      shift(@active_argv);
      next;
      }
    ($first,$rest) = ($1,$2);

    $pos = index($optlist, $first);
    if ($pos < 0) {
      print STDERR "Unknown option: $first\n";
      ++$opt_err;
      if($rest ne '') {
	$active_argv[0] = "-$rest";
	}
      else {
	shift(@active_argv);
	}
      next;
      }

    if (!defined($args[$pos + 1]) ||
	$args[$pos + 1] ne ':') { # simple switch argument
      eval "\$opt_$first = 1";
      if ($rest eq '') {
	shift(@active_argv);
	}
      else {
	$active_argv[0] = "-$rest";
	}
      next;
      }

    # switch takes argument
    if ($rest ne '') {
      eval "\$opt_$first = \"$rest\"";
      shift(@active_argv);
      next;
      }

    shift(@active_argv);
    if ($rest eq '') { # argument is $active_argv[0]
      if (!@active_argv) {
	print STDERR "-$first: argument missing\n";
	$opt_err ++;
	last;
	}
      $rest = shift(@active_argv);
      if ($rest eq "--") {
	print STDERR "-$first: argument missing\n";
	$opt_err ++;
	last;
	}
      } # argument is $active_argv[0]
    eval "\$opt_$first = \$rest;";
    } # process argv

  # Remaining args are left for caller...
  @ARGV = (@new_argv, @unused_argv);;
} # Getopts

#-----------------------------------------------
# Find the first line that matches the given header and return it.
sub extract_header {
  local($pattern) = @_;
  local($result);

  $result = "";
  for (@original) {
    last if length($_) == 1; # RFC-822: end of headers at first blank line
    if ($result ne "") {
      last if $_ !~ /^\s+(.*)/;	# not a continued header, and we're done.
      $result .= " "; 
      $result .= $1;
      chop($result);
      }
    elsif ($_ =~ /$pattern/i) { # RFC-822 says case-independent
      $result = $_;
      chop($result);
      $result =~ s/$pattern\s*(.*)/$1/i;
      }
    }
  return $result;
} # extract_header

#-----------------------------------------------
# Return true if the given pattern is a private (intra-net) IP
# (from RFC 1597)
# ---------
# 3. Private Address Space 
#    
#    The Internet Assigned Numbers Authority (IANA) has reserved the
#    following three blocks of the IP address space for private networks:
#    
#         10.0.0.0        -   10.255.255.255
#         172.16.0.0      -   172.31.255.255
#         192.168.0.0     -   192.168.255.255

sub is_private_inet_addr {
  local($pat) = @_;

  return 0 unless $pat =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/;
  return 1 if $1 == 10;
  return 1 if $1 == 172 && $2 >= 16 && $2 <= 31;
  return 1 if $1 == 192 && $2 == 168;
  return 0;
  }

#-----------------------------------------------
# Return true if the given pattern is an inet address, e.g. "192.83.233,4"
sub is_inet_addr {
  local ($pat) = @_;

  return 0 unless $pat =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/;
  return 0 if $1 > 255;
  return 0 if $2 > 255;
  return 0 if $3 > 255;
  return 0 if $4 > 255;
  return 1;
  }

#-----------------------------------------------
# return the connecting SMTP agent's name for itself
sub helo_name {
  local ($text) = @_;
  local ($probe);

  return "" unless $text =~ /.*from\s*[\w\.-]+.*/;
  ($probe = $text) =~ s/.*from\s*([\w\.-]+).*/$1/;	# get first word

  $probe =~ s/^\((.*)\).*/$1/;	# turn (host) into host
  $probe =~ s/(.*)\(.*\).*/$1/;	# turn host(inet_addr) into host
  $probe =~ s/^(\S+)\s+.*/$1/;	# get first word
  $probe = &good_hostname($probe, 0);	# don't require this name to exist
  return "" if $probe eq "";

  return $probe;
  }

#-----------------------------------------------
# Return name of recipient machine ("by" clause in "Received:" header)
sub by_name {
  local ($text) = @_;
  local ($probe);

  return "" unless $text =~ /.*by\s+([\w\.-]+).*/;
  $probe = &good_hostname($1, 0);	# peer must exist!
  return $probe;
  }

#-----------------------------------------------
# Return (name, inet_addr) of peer name out of received header if they exist
sub received_peer_name {
  local ($text) = @_;
  local ($probe, $probe2);
  local ($peer_name, $inet_addr);

  return ("", "") unless $text =~ /.*from\s*[^(]*\([^)]*\)/;

  $probe = $text;

  # a _very_ thorny problem.  Extract the from-clause (stuff between parens).
  # Ignore any by-clause.
  if ($probe =~ /\(peer crosschecked as: (.*)\)/)  {
    # new stuff from uunet, yay!  Sep '97, they're finally hitting back.
    $probe = $1;
    }
  else {
    $probe =~ s/.*\bfrom\s+(.*)\s+by\s.*/$1/;  # by-clause after from-clause
    $probe =~ s/.*\bby\s+.*\s+from\s+(.*)/$1/;  # from-clause after by-clause
    return "" unless $probe =~ /[^(]*\(([^)]*)\).*/; # must have parens here.
    $probe = $1;			# stuff between parens
    }

  $inet_addr = $probe;
  $probe =~ s/\s*(\S+).*/$1/; # grab first word
  return ("", "") if $probe =~ /.*smail.*/i;  # strange smail header
  return ("", "") if $probe =~ /.*smtp.*/i;  # strange smtpd header
  return ("", "") if $probe =~ /^\d\.\d/;  # strange sendmail header

  # Get the internet address.
  $inet_addr =~ s/^[^[]*\[(.*)\]/$1/;
  $inet_addr = "" if !&is_inet_addr($inet_addr);

  $probe =~ s/^([^[]*)\[.*\]/$1/; # ignore bracketed inet address
  if (&is_inet_addr($probe)) { # mailhost was as puzzled as we are
    $inet_addr = $probe;
    $probe = "";
    }

  $probe = &good_hostname($probe, 0);
  if ($probe eq "localhost" || $inet_addr eq "127.0.0.1") {
    $probe = $inet_addr = "";
    }
  return ($probe, $inet_addr);
  }

#-----------------------------------------------

sub dict_at {
  local($pos) = @_;
  local($b1, $b2, $b3, $b4);

  $pos *= 4;
  $b1 = ord(substr($EXEMPT_LIST, $pos, 1));
  $b2 = ord(substr($EXEMPT_LIST, $pos + 1, 1));
  $b3 = ord(substr($EXEMPT_LIST, $pos + 2, 1));
  $b4 = ord(substr($EXEMPT_LIST, $pos + 3, 1));

  return ($b1 << 24) | ($b2 << 16) | ($b3 << 8) | $b4;
  }

sub dawg_check {
  local($word) = @_;
  local($edgePos);
  local($wordChar, $wordPos);

  $edgePos = 1;			# root node

  $wordPos = 0;
  $wordChar = ord(substr($word, $wordPos, 1));
  for (;;) {
    if ($wordChar == ((&dict_at($edgePos) >> 24) & 0xff)) {
				# character match
      $wordPos ++;		# extract next char in word
      $wordChar = ord(substr($word, $wordPos, 1));
      $wordChar = 0 if !defined($wordChar);
      if ($wordChar == 0) {	# end of word
	return (&dict_at($edgePos) & 0x800000) != 0;
        }
      else {			# jump to new node
        $edgePos = &dict_at($edgePos) & 0x1ffff;
	last if $edgePos == 0;	# null pointer, we're done
	next;
	}
      }
    last if (&dict_at($edgePos) & 0x400000) != 0;	# end of node, no match
    $edgePos ++;		# next entry in this node
    }
  return 0;			# not found
  }

#-----------------------------------------------
# Return 1 if the given name is in the list of exempt sites.
sub is_exempt {
  local ($name) = @_;
  local ($site);

  return 0 if $name eq "";
  $name =~ y/A-Z/a-z/;		# canonicalize to lower case
  $site = &find_site($name);	# site, not host.
#  $site = $name;

  return 0 if !&dawg_check($site);
  print "found exempt domain \"$site\".\n" if $verbose;
  return 1;
}

#-----------------------------------------------
# Return 1 if arg begins with a valid month (RFC-822)
sub is_month {
  local ($text) = @_;
  local ($probe, $each, $result);

  ($probe = $text) =~ s/^\s*(\S\S\S)\s+.*/$1/;
  for (@monthNames) {
    return 1 if $_ =~ /$probe/i;
    }
  return 0;
}

#-----------------------------------------------
# Return 1 if the arg is a legal (RFC-822) day of the week.
sub is_day {
  local ($text) = @_;
  local ($probe, $result);

  ($probe = $text) =~ s/^\s*(\S\S\S).*/$1/;
  $probe =~ s/(\W)/\\$1/g;		# remove special characters
  for (@wdayNames) {
    return 1 if $_ =~ /$probe/i;
    }
  return 0;
  }

#-----------------------------------------------
# Is time zone believable?
sub zone_check {
  local ($datepart) = @_;
  local ($hour_offset, $zone, $zoneStr);
  local (@zone_list, $each);

  $hour_offset = $zone = $datepart;
  if ($hour_offset =~ /(-\d\d\d\d)/) {
    $hour_offset = $1;
    }
  elsif ($hour_offset =~ /\+(\d\d\d\d)/) {
    $hour_offset = $1;
    }
  else {
    $hour_offset =~ s/^.*\D(\d\d\d\d).*/$1/;
    }
  return 1 if $hour_offset eq $datepart;

  # $pattern = "^.*" . $hour_offset . "\s*\(([^)]+)\).*";
  # $zone =~ s/$pattern/$1/;
  $zone =~ s/^.*$hour_offset\s*\(([^)]+)\).*/$1/;
  return 1 if $zone eq $datepart;

  $zoneStr = $timezones{$hour_offset};
  if ($zoneStr eq "") {
    print
      "zone_check:  Unknown time zone \"$hour_offset\" in date \"$datepart\"\n"
	if $verbose;
    $header_analysis .=
	"${bullet}I cannot confirm that the time zone \"$zone\" in \"Received:\"\n";
    $header_analysis .= "header \"$datepart\"\n";
    $header_analysis .= "should have an hour offset of \"$hour_offset\".\n\n";
    return 1;
    }
  @zone_list = split(/,\s*/, $zoneStr);
  for (@zone_list) {
    return 1 if $_ =~ /^$zone$/i; # OK, let it be case-insensitive
    }

  if ($verbose) {
    print "zone_check: hour offset $hour_offset has zone $zone, ";
    print "expected one of: $zoneStr\n";
    }

  # prettify for the message
  $zoneStr = "";
  for (@zone_list) {
    $zoneStr .= ", " if $zoneStr ne "";
    $zoneStr .= "\"$_\"";
    }
  if ($zoneStr =~ /^[^,]+,[^,]+$/) {
    $zoneStr =~ s/(.*), ([^,]*)$/$1 or $2/g;
    }
  else {
    $zoneStr =~ s/(.*), ([^,]*)$/$1, or $2/g;
    }

  $header_analysis .= "${bullet}Time zone in \"Received:\" header \"$datepart\"\n";
  $header_analysis .=
    "is \"$zone\" when it should be one of: $zoneStr.\n";
  $header_analysis .=
	"(This is the signature of the \"Stealth\" spam mailer.)\n"
	if $zone eq "EST";
  $header_analysis .= "\n";

  return 0;
}


#-----------------------------------------------
# Return 1 if the header parses properly
sub parse_received_header {
  local ($text) = @_;
  local ($datepart, $probe, $probe2, $each);

# From rfc822:
# TO DO:  note that "from" must be first, etc.
#     received    =  "Received"    ":"            ; one per relay
#                       ["from" domain]           ; sending host
#                       ["by"   domain]           ; receiving host
#                       ["via"  atom]             ; physical path
#                      *("with" atom)             ; link/mail protocol
#                       ["id"   msg-id]           ; receiver msg id
#                       ["for"  addr-spec]        ; initial form
#                        ";"    date-time         ; time received
#     date        =  1*2DIGIT month 2DIGIT        ; day month year
#                                                 ;  e.g. 20 Jun 82
#     date-time   =  [ day "," ] date time        ; dd mm yy
#                                                 ;  hh:mm:ss zzz
#     day         =  "Mon"  / "Tue" /  "Wed"  / "Thu"
#                 /  "Fri"  / "Sat" /  "Sun"
#     hour        =  2DIGIT ":" 2DIGIT [":" 2DIGIT]
#                                                 ; 00:00:00 - 23:59:59
#     month       =  "Jan"  /  "Feb" /  "Mar"  /  "Apr"
#                 /  "May"  /  "Jun" /  "Jul"  /  "Aug"
#                 /  "Sep"  /  "Oct" /  "Nov"  /  "Dec"
#     time        =  hour zone                    ; ANSI and Military
#     zone        =  "UT"  / "GMT"                ; Universal Time
#                                                 ; North American : UT
#                 /  "EST" / "EDT"                ;  Eastern:  - 5/ - 4
#                 /  "CST" / "CDT"                ;  Central:  - 6/ - 5
#                 /  "MST" / "MDT"                ;  Mountain: - 7/ - 6
#                 /  "PST" / "PDT"                ;  Pacific:  - 8/ - 7
#                 /  1ALPHA                       ; Military: Z = UT;
  if ($text !~ /;/) {
    $header_analysis .= "${bullet}\"Received:\" header \"$text\"\n";
    $header_analysis .= "has no mandatory semicolon (see RFC-822).\n\n";
    return 0;
    }

  # But does it really have a time?
  ($datepart = $text) =~ s/.*;\s*(.*)/$1/;
  $probe = $datepart;

  # Remove comments (thanks to Andrey A. Chernov for this)
  $probe =~ s/(\([^)]*\))//g;

  if ($probe =~ /,/) {
    # process the day field
    ($probe2 = $probe) =~ s/^\s*([^,\s]+)\s*,.*/$1/;
    if (!&is_day($probe)) {
      $header_analysis .=
	  "${bullet}Day of week in \"Received:\" header, \"$datepart\",\n";
      $header_analysis .=
	  "is not standard (see RFC-822).\n\n";
      return 0;
      }
    $probe =~ s/[^,\s]+\s*,(.*)/$1/;  # now looking at date + time
    }
  else {
    # Nonstandard netcom header
    $probe =~ s/^\s*...\s+(.*)/$1/ if &is_day($probe);
    }

  $probe =~ s/^\s*(.*)/$1/; # remove leading spaces

  if ($probe =~ /^\d+\s+(.*)/) { # standard day-month-year
    $probe = $1;		# now must have month

    if (!&is_month($probe)) {
      $header_analysis .=
	"${bullet}Date in \"Received:\" header, \"$datepart\", is not\n";
      $header_analysis .= "standard (see RFC-822).\n\n";
      return 0;
      }
    $probe =~ s/^...\s+(.*)/$1/;
    }
  elsif (&is_month($probe)) {
    # Netcom has non-standard month-day-year
    $probe =~ s/^...\s+(.*)/$1/;
    if ($probe !~ /^\d/) {
      $header_analysis .= "${bullet}(Netcom-style) ";
      $header_analysis .=
	  "date in \"Received:\" header, \"$datepart\", is not standard;\n";
      $header_analysis .= "cannot find day of month (see RFC-822).\n\n";
      return 0;
      }
    }
  else {
    $header_analysis .=
	"${bullet}Date in \"Received:\" header, \"$datepart\", is not\n";
    $header_analysis .= "standard; cannot find day of month (see RFC-822).\n\n";
    return 0;
    }

  # Now for year
  $probe =~ s/^...\s+(.*)/$1/;
  if ($probe !~ /^\d+/) {
    $header_analysis .=
	"${bullet}Date in \"Received:\" header, \"$datepart\", is not\n";
    $header_analysis .= "standard; cannot find the year (see RFC-822).\n\n";
    return 0;
    }
  $probe =~ s/^\d+\s+(.*)/$1/;  # anything left?
  if ($probe !~ /^\d\d:\d\d/) {
    $header_analysis .=
	"${bullet}Date in \"Received:\" header, \"$datepart\", is not\n";
    $header_analysis .= "standard; time is not hh:mm:ss (see RFC-822).\n\n";
    return 0;
    }

  # in practice, time zones are not very consistent, but we'll do our best.
  return 0 if !&zone_check($datepart);

  return 1;
  }

#-----------------------------------------------
# Indicate if the Received: header is forged.
sub forged_received_header {
  local ($text) = @_;
  local ($possibleName);
  local ($probe, $probe2, $result);

  # Many ad hoc checks for a bogus Received: header.
  # Plenty of work needs to be done here.
  # See http://oasis.ot.com/~dmuth/spam-l

  # everything after the 'from'
  ($possibleName = $text) =~ s/^Received:\s*(.*)/$1/i;

  $result = 0;

  for ("CLOAKED!", "may be forged", "unknown host") {
    next unless $possibleName =~ /$_/i;
    print "forged_received_header: found \"$_\" in header.\n" if $verbose;
    $header_analysis .=
	"${bullet}\"Received:\" header has suspicious text: \"$_\".\n\n";
    $result = 1;
    }

  # look for impossible IP address
  $probe2 = $probe = $possibleName;
  for (;;) {
    local ($str1, $str2, $str3, $str4);

    # last if !($probe =~ /\d+\.\d+\.\d+\.\d+/);
    last unless $probe =~ /[\[(]\s*(\d+\.\d+\.\d+\.\d+)\s*[\])]/;
    # idea: require that the inet_addr have enclosing brackets?

    ($str1 = $probe2 = $1) =~ s/\./\\./g;
    $probe =~ s/^(.*)$str1(.*)/$1$2/;	# remove it

    ($str1, $str2, $str3, $str4) = ($probe2 =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)/);

    if ($probe2 =~ /^0\d/ || $probe2 =~ /\.0\d/ ||
      $str1 > 255 || $str2 > 255 || $str3 > 255 || $str4 > 255) {
      print "forged_received_header: impossible IP \"$probe2\".\n" if $verbose;
      $header_analysis .=
	  "${bullet}\"Received:\" header has impossible IP \"$probe2\".\n\n";
      $result = 1;
      }
    }

  $result = 1 if !&parse_received_header($possibleName);
  # bogus Received: header won't have a for-clause with _your_ name in it.
  # for now, just see if it has the right domain name.
  # I wish a Received: header _always_ had a for-clause!
# too aggressive.  How can I make this right?
#    if ($possibleName =~ /for\s+<[^>]+>;/) {
#      $probe = $possibleName;
#      $probe =~ s/.*\sfor\s+<([^>]+)>;.*/$1/;
#      if (!&home_domain($probe)) {
#	print 
#	    "forged_received_header: bogus for-clause <$probe>\n"
#		if $verbose;
#	$header_analysis .= "${bullet}\"Received:\" header has bogus for-clause <$probe>.\n\n";
#	$result = 1;
#        }
#      }

  # bogus Received: header often has incorrect SMTP id
# Relies too much on the implementation of SMTP.
#    if ($possibleName =~ /with smtp id/i) {
#      local ($smtp_id, $smtp_char, $smtp_time, $smtp_hour);
#
#      $smtp_id = $possibleName;
#      $smtp_id =~ s/.*with smtp id (\w+).*/$1/i;
#      $smtp_char = $smtp_id;
#      $smtp_char =~ s/^(.).*$/$1/;
#      $smtp_time = $possibleName;
#      $smtp_time =~ s/.*with smtp id [^;]+;(.*)/$1/i; # entire end of header
#      $smtp_time =~ s/.*\s([0-2]\d:\d\d:\d\d).*/$1/;  # just want hh:mm:ss
#      $smtp_hour = $smtp_time;
#      $smtp_hour =~ s/^([0-2]\d):.*/$1/;	# just the hour
#      if ($smtp_chars{$smtp_char} ne $smtp_hour) {
#	if ($verbose) {
#	  print "forged_received_header: SMTP id \"$smtp_id\" does";
#	  print "    not match time \"$smtp_time\"\n";
#	  }
#	$header_analysis .=
#"${bullet}\"Received:\" header with SMTP id \"$smtp_id\" mismatches \"$smtp_time\".\n\n";
#	$result = 1;
#	}
#      }

  return $result;
  }

#-----------------------------------------------
# Find a line in hosts
sub findLine {
  local ($text) = @_;
  local ($result, $line);

  return "" if $hosts_file eq "";
  open(HOSTS, "<$hosts_file") || die "cannot open $hosts_file: $!";
  $result = "";
  while (<HOSTS>) {
    $line = $_;
    chop($line);
    next if $line =~ /^\s*#.*/;
    # how else do I specify \< and \>?  I miss vi's regex, for once.
    if ($line =~ /^$name$/i ||
        $line =~ /^$name\s+/i ||
        $line =~ /\s+$name$/i ||
        $line =~ /\s+$name\s+/i) {
      $result = $line;
      last;
      }
    next;
    }

  close HOSTS;
  return $result;
  }

#-----------------------------------------------
# search the hosts file.  addrToName looks here first
sub fakeAddrToName {
  local($name) = @_;
  local($result);

  $name =~ s/(\W)/\\$1/g;		# remove special characters
  $result = &findLine($name);

  $result =~ s/^\d+\.\d+\.\d+\.\d+\s+(\S+).*/$1/;
  return $result;
  }

#-----------------------------------------------
# search the hosts file.  nameToAddr looks here first
sub fakeNameToAddr {
  local($name) = @_;
  local($result);

  $name =~ s/(\W)/\\$1/g;		# remove special characters
  $result = &findLine($name);
  $result =~ s/^(\d+\.\d+\.\d+\.\d+)\s+.*/$1/;
  return $result;
  }

#-----------------------------------------------
# Cache the relationship between the given hostname and IP
sub log_mapping {
  local($name, $ip, $type) = @_;

  $ips_to_hosts{$ip} = $name if $ip ne "";
  $hosts_to_ips{$name} = $ip if $name ne "";
  $name = "?" if $name eq "";
  $ip = "?" if $ip eq "";
  print "  $ip -> $name\t# $type\n" if $verbose;
  }


#-----------------------------------------------
# Try to turn an address into a hostname (cover for gethostbyaddr)
sub addrToName {
  local ($text) = @_;
  local ($name, $aliases, $addrtype, $length, @addrs);
  local ($addr_int);
  local ($a, $b, $c, $d);

  $name = $ips_to_hosts{$text};
  return $name if defined($name) && $name ne "";

  $name = &fakeAddrToName($text);
  if ($name ne "") {
    $name = "" if $name eq "BOGUS";
    &log_mapping($name, $text, "hosts");
    return $name;
    }

  ($a, $b, $c, $d) = split(/\./, $text);
  $addr_int = pack("C4", $a, $b, $c, $d);

  ($name, $aliases, $addrtype, $length, @addrs) =
	gethostbyaddr($addr_int, &AF_INET);
  if (defined($name) && $name ne "") {
    &log_mapping($name, $text, "gethostbyaddr");
    return $name;
    }

  &log_mapping("", $text, "gethostbyaddr");
  return "";
  }

#-----------------------------------------------
# Try to turn a hostname into an address (cover for gethostbyname)
sub nameToAddr {
  local ($text) = @_;
  local ($name, $aliases, $addrtype, $length, @addrs);
  local ($a, $b, $c, $d);

  $name = $hosts_to_ips{$text};
  return $name if defined($name) && $name ne "";

  $name = &fakeNameToAddr($text);
  if (defined($name) && $name ne "") {
    $name = "" if $name eq "0.0.0.0";
    &log_mapping($text, $name, "hosts");
    return $name;
    }

  ($name, $aliases, $addrtype, $length, @addrs) = gethostbyname($text);
  if (defined($name) && $name ne "") {
    ($a, $b, $c, $d) = unpack('C4', $addrs[0]);
    $name = "$a.$b.$c.$d";
    &log_mapping($text, $name, "gethostbyname");
    return $name;
    }

  &log_mapping($text, "", "gethostbyname");
  return "";
  }

#-----------------------------------------------
# return 1 if this hostname resolves
sub real_host {
  local ($host) = @_;
  local ($junk);

  return 1 if $dont_resolve;
  $junk = &nameToAddr($host);
  return 0 if $junk eq "";
  return 1;
  }


#-----------------------------------------------
# return "", or best domain from the "Received: from" headers
# The algorithm is fairly involved.  Basically, start with the newest
# Received: header and process until the last one a bogus one is found.
sub do_received {
  local ($message_host) = @_;
  local ($message_site, $inet_addr);
  local ($result, $previous_peer);
  local ($probe, $probe2, $possibleName);
  local ($this, $i, $each);
  local (@all_headers);
  local ($peer_name, $helo_name, $by_name);
  local ($peer_site, $helo_site, $by_site);
  local ($junk2, $peer_is_exempt, $found_public_addr);

  $message_site = &find_site($message_host);

  # get "Received: " headers for later perusal.
  for ($i = 0; $i < @original; $i ++) {
    $line = $original[$i];
    last if length($line) == 1; # RFC-822: end of headers at first blank line
    next unless $line =~ /^Received:\s*/i;	# RFC-822 says case-independent
    $this = $line;
    chop($this);
    while ($i < @original) { # collect multi-line header
      $line = $original[$i + 1];
      last unless $line =~ /^[\s]+/;
      chop($line);
      $this .= $line;
      $i ++;
      }
    $this =~ s/^Received:\s*(.*)/$1/i;
    last if &forged_received_header($this); # don't _even_ put in the list
    push(@all_headers, $this);
    }

  $peer_name = $result = "";
  $found_public_addr = 0;

  # Analyze the headers
  NEXT_HEADER: for ($i = 0; $i < @all_headers; $i ++) {
    $possibleName = $all_headers[$i];

    $previous_peer = $peer_name if $peer_name ne "";
    ($peer_name, $inet_addr) = &received_peer_name($possibleName);
    $helo_name = &helo_name($possibleName);
    $by_name = &by_name($possibleName);
    if ($verbose) {
      print "received:  \"$peer_name\"";
      print " inet: \"$inet_addr\"" if defined($inet_addr) && $inet_addr ne "";
      print " by: \"$by_name\"" if $by_name ne "";
      print " helo: \"$helo_name\"" if $helo_name ne "";
      print "\n";
      }
    if (defined($inet_addr) && $inet_addr ne "") {
      local($is_private) = &is_private_inet_addr($inet_addr);

      if (!$is_private) {
	# we're in the part of the received: path that was on the Internet
        $found_public_addr = 1;
        }
      elsif ($found_public_addr) {
	# we've fallen into the private intranet on the other end.  Stop.
	print "  do_received: found sender's intranet $inet_addr\n"
	  if $verbose;
	last NEXT_HEADER;
	}
      else {
	# we're in the private intranet at the receiving end.  Skip.
	print "  do_received: found receiver's intranet $inet_addr\"\n"
	  if $verbose;
	next NEXT_HEADER;
	}
      }
    $peer_site = &find_site($peer_name);
    &good_hostname($peer_name, 1);	# adds complaint if can't confirm
    $peer_is_exempt = &is_exempt($peer_name);
    $peer_site = "" if $peer_is_exempt;

    $helo_site = &find_site($helo_name);
    # &good_hostname($helo_name, 1);	# adds complaint if can't confirm

    $by_site = &find_site($by_name);
    &good_hostname($by_name, 1);	# adds complaint if can't confirm

    # analyze inet_addr, upgrade notion of peer_name
    if (defined($inet_addr) && $inet_addr ne "" && !$dont_resolve) {
      $probe2 = &addrToName($inet_addr);
      if ($peer_name ne "" && $probe2 eq "") {
	$header_analysis .=
	    "${bullet}Unable to confirm $peer_name is $inet_addr.\n\n";
	$peer_name = $previous_peer if $peer_name eq "";
	}
      elsif ($peer_name ne "" && $probe2 ne "" && !&home_domain($peer_name)) {
	$junk2 = &find_site($probe2);
	if (!$peer_is_exempt && $peer_site ne $junk2) {
	  $header_analysis .= "${bullet}\"Received:\" header says that peer is \"$peer_name\",\n";
	  $header_analysis .= "but address $inet_addr resolves to $probe2.\n\n";
	  $result = $probe2;
	  last NEXT_HEADER;
	  }
	}
      if ($probe2 ne "") {
	if ($peer_name eq "") {
	  $header_analysis .= 
	      "${bullet}\"Received:\" header has empty peer name, but\n";
	  $header_analysis .= "address $inet_addr resolves to $probe2.\n\n";
	  }
	$peer_name = $probe2;
	}
      } # $inet_addr ne ""

    # ignore this header if it comes through/from/by a exempt domain
    if ($peer_is_exempt) {
      $result = $by_name if $by_name ne "";
      last NEXT_HEADER;
      }
    $peer_site = &find_site($peer_name);

    # Look at helo_name.  Don't trust it very far, really.
    if ($helo_name ne "" && $helo_site eq "") {
      $header_analysis .=
	  "${bullet}HELO name \"$helo_name\" is not a valid host name.\n\n";
      print "  do_received: queer helo_name $helo_name\"\n" if $verbose;
      }
    elsif (&is_exempt($helo_name)) {
      $result = $peer_site if $peer_site ne "";  # end of the road
      last NEXT_HEADER;
      }
    elsif ($helo_name ne "" && $helo_name ne $peer_name) { # analyze helo_name
      # Does the helo_name matches the peer_name?
      if ($peer_site ne "" && $helo_site ne $peer_site) {
	$header_analysis .=
	  "${bullet}\"Received:\" header's HELO name \"$helo_name\" significantly\n";
	$header_analysis .= "differs from peer_name \"$peer_name\".\n\n";
	print "  do_received: helo $helo_name != peer $peer_name\n" if $verbose;
        $result = $peer_site;  # end of the road
	last NEXT_HEADER;
	}
      } # analyze helo_name

    # Attempt to match the name of this host with the by_name of the next header
    if ($helo_name ne "" &&
	$i != $#all_headers) { # not oldest header, verification possible
      # Attempt to match the helo_name with the by_name of the next header
      $probe = &by_name($all_headers[$i + 1]);
      $probe2 = &find_site($probe);
      if ($probe2 eq "") {
	print
	    "adcomplain: unable to confirm $helo_name against origination.\n";
	}
      elsif ($helo_site ne "" && $probe2 ne $helo_site) {
	$header_analysis .=
	   "${bullet}HELO name \"$helo_name\" does not match prior header's\n";
	$header_analysis .= "by-name \"$probe\"\n\n";
	print
	    "do_received: helo $helo_name does not match next by_name $probe\n"
	    if $verbose;
	# HELO name is a little queer, so tolerate this, with a comment.
	}
      elsif ($peer_site ne "" && $probe2 ne $peer_site) {
	$header_analysis .=
	   "${bullet}Peer name \"$peer_name\" does not match prior header's\n";
	$header_analysis .= "by-name name \"$probe\"\n\n";
	print
	    "do_received: peer $peer_name does not match by_name $probe\n"
	    if $verbose;
	$result = $peer_site;
	last NEXT_HEADER;	# This is serious!
	}
      } # not oldest header

    # Check for spam insertion that created a Message-Id
    if ($result ne "" && $result eq $message_site &&
	$peer_site ne "" && $peer_site ne $message_site) {
      # No valid message has a Message-Id created on it mid-stream.  If we
      # get here, we have the spam insertion.  Let the victimized postmaster
      # be held accountable.
      $header_analysis .=
	  "${bullet}Message-Id site \"$message_site\" is not at the start"
		. " of the chain of Received: headers.\n\n";
      print 
	"do_received: Message-Id site \"$message_site\" found.\n" if $verbose;
      $result = $peer_site if $peer_site ne "";
      last NEXT_HEADER;
      }

    # $peer_name is definitely the way to go:
    if ($peer_name ne "") {
      $result = $peer_site;
      next NEXT_HEADER;
      }

    # no reverse address.  Could be bogus, DNS transient oddness, or
    # simply that we are behind a firewall.

    # Plan B: use the by-name
    last NEXT_HEADER if &is_exempt($by_name);
    if ($by_name ne "") {
      $result = $by_name;
      next NEXT_HEADER;
      }

    # Go for the helo name (yuccch)
    next NEXT_HEADER if $helo_name eq "";
    last NEXT_HEADER if &is_exempt($helo_name);

    print "do_received: relying on helo_name \"$helo_name\"\n" if $verbose;
    $result = $helo_name;
    } # NEXT_HEADER

  return "" if $result eq "";
  $result = "" if &home_domain($result);
  if ($result ne "") {
    &register_host("Received:", $result);
    print "do_received: result is \"$result\"\n";
    }

  if ($result ne "" && $message_host ne "") {
    $probe = &find_site($result);
    if ($message_site ne $probe) {
      $header_analysis .=
	  "${bullet}Site in \"Message-Id:\" \"$message_site\" does not match\n";
      $header_analysis .=
	  "\"Received:\" origin \"$result\".\n\n";
      print "  do_received: Message-Id site $message_site != result $result\n"
	  if $verbose;
      }
    }

  return $result;
  }

#-----------------------------------------------
# The following headers are scanned, in order, to determine the poster:
#   NNTP-Posting-User:
#   Return-Path:
#   Reply-To:
#   From:
#
# A poster's address be of the form user@domain, where domain is
# a legal domain name.  The domain is rejected if it matches our domain.
#
# If the address is of form "user,id <xxx@site>", the
# message will be addressed to xxx@site.
# If the address is of form "Name <address>", set to "address".
# If the address is of form "(address)", set to "address".
# If the address is of form "address(Name)", set to "address".
# Remove extra words, in case header is of form "address (Name)"

sub find_poster {
  local ($this, $junk, $junkhost, $result);
  local (@headers, @search_patterns);

  @headers = ("NNTP-Posting-User:", "Return-Path:", "Reply-To:", "From:");
  @search_patterns = ( '.*NNTP-Posting-User: ',
      '^Return-Path:', '^Reply-To:', '^From:');
  $result = "";
  for (@search_patterns) {
    $this = $headers[0];
    shift @headers;
    $junk = &extract_header($_); 
    next if $junk eq "";

    # Validate the name
    $junk =~ s/[^<]*<([^>]*)>.*/$1/;		# remove angle brackets
    $junk =~ s/([^(\s]*)\s*\([^)]*\)/$1/;	# remove parenthesized name
    ($junkhost = $junk) =~ s/.*@(.*)/$1/;
    $junkhost =~ s/([^\s]*) *\(.*\)/$1/;	# remove proper name
    $junkhost = &good_hostname($junkhost, 0);
    next if $junkhost eq "";
    $result = $junk if $result eq "";	# take first available
    &register_host($this, $junkhost);
    print "find_poster: found '$this' host $junk\n" if $verbose;
    }
  if ($result !~ /.*,.*/) {
    # remove angle brackets from poster's name
    $result =~ s/.*<(.*)>.*/$1/;
    }
  if ($result eq "") {
    print "\nadcomplain: WARNING: cannot determine poster's address.\n\n";
    $result = "";
    }
  else {
    print "find_poster: poster is '$result'\n" if $verbose;
    }
  return $result;
  }

#-----------------------------------------------
# Analyze the "Path:" header, return best possible host
# Basically, we're looking for the oldest (believable) host in the path.
# I used to process the path from left-to-right, with the idea that I
# could stop before the first unconfirmable host.  In practice, it seems
# to be better to just go from the right and stop with the first believable
# one (though I still have reservations about that).
sub do_path {
  local ($path, $result, $chunk, $junk);
  local (@chunks);

  $path = &extract_header("^Path:");
  return "" if $path eq "";
  @chunks = reverse(split(/!/, $path));

  foreach $chunk (@chunks) {
    $chunk = &addrToName($chunk) if &is_inet_addr($chunk);
    $chunk = &good_hostname($chunk, 1);
    # $chunk = &good_hostname($chunk, 0);	# don't require host to exist.
    next if $chunk eq "";
    next if &is_exempt($chunk);
    $result = $chunk;
    last;
    }

  if ($result ne "") {
    print "do_path: Path origin is $result\n" if $verbose;
    &register_host("Path:", $result);
    return $result;
    }
  return "";
  }

#-----------------------------------------------
# Examine and return a hostname based on given header
sub try_host {
  local($pat, $name) = @_;
  local($junk, $junk2);

  $junk = &extract_header($pat);
  return "" if $junk eq "";
  print "try_host: \'$name\' is $junk\n" if $verbose;
  if (&is_inet_addr($junk) && !$dont_resolve) {
    $junk2 = &addrToName($junk);
    $junk = $junk2;
    }
  else {
    $junk = &good_hostname($junk, 0);
    }
  return "" if $junk eq "";
  return "" if &home_domain($junk);
  &register_host($name, $junk);
  return $junk;
}

#-----------------------------------------------
# I've seen these headers and I want to check for them:
# X-Authentication-Warning: quick.t-1net.com: b142.ecom.net [207.13.225.142] didn't use HELO protocol
#X-Authentication-Warning: communique-tech.co.uk: 
#	Host 1Cust40.max15.los-angeles.ca.ms.uu.net [153.34.77.168] claimed 
#	to be hotmail.com
# X-Authentication-Warning: gatekeeper.smltd.com: 
#	mail set sender to <lori@6t9.com> using -f
# X-Authentication-Warning: ns.internetmedia.net: 
#	weginc owned process doing -bs
sub do_auth_warning {
  local ($junk, $junk_site, $name_site);
  local ($name, $inet_addr, $result);

  $junk = &extract_header("X-Authentication-Warning:");
  return "" if $junk eq "";

  $inet_addr = "";
  if ($junk =~ /didn't use HELO protocol/) {
    $junk =~ s/^[^:]+: (.*) didn't use HELO protocol/$1/i;
    $name = $inet_addr = $junk;

    $name =~ s/^\s*([\w\.-]+)\s*.*/$1/;
    $inet_addr =~ s/^.*\[(.*)\].*/$1/;	# remove bracketed name
    $name =~ s/^(.*)\[.*\].*/$1/;	# remove bracketed name
    }
  elsif ($junk =~ /claimed to be/) {
#	Host 1Cust40.max15.los-angeles.ca.ms.uu.net [153.34.77.168] claimed 
# or:
#	oncidium: Host x.y..net claimed to be usinet cziegle

    ($name = $junk) =~ s/.*Host\s+(\S+).*/$1/;

    ($inet_addr = $junk) =~ s/.*\[([^]]*)\].*/$1/
	if $junk =~ /\[\d+\.\d+\.\d+\.\d+\]/;
    }
  elsif ($junk =~ / set sender to /) {
    # nothing else warranted
    $name = "";
    }
  elsif ($junk =~ / owned process doing /) {
    # is this useful?  For now, do nothing.
    $name = "";
    }
  else {
    print "adcomplain: new X-Authentication-Warning header\n";
    print "  Please send this test case to the adcomplain authors.\n";
    return "";
    }

  $result = $name;
  if ($inet_addr ne "" && !$dont_resolve) {
    $junk = &addrToName($inet_addr);
    if ($name ne "" && $junk eq "") {
      $header_analysis .= "${bullet}Unable to confirm $name is $inet_addr.\n\n";
      }
    elsif ($name ne "" && !&home_domain($name)) {
      $result = $junk;
      $junk_site = &find_site($junk);
      $name_site = &find_site($name);
      if ($name_site ne $junk_site) {
	$header_analysis .= "${bullet}\"X-Authentication-Warning:\" header says that peer is \"$name\",\n";
	$header_analysis .= "but address $inet_addr resolves to $junk.\n\n";
	}
      }
    } # $inet_addr ne ""
  print "do_auth_warning: found \"$result\"\n" if $verbose;
  &register_host("X-Authentication-Warning:", $result) if $result ne "";
  return $result;
  }


#-----------------------------------------------
# The following order of preference is used to determine the originating host:
#   the host in the "NNTP-Posting-Host:" header;
#   the host in the "Path:" header;
#   the host in the "Received:" headers;
#   the host in the "Message-Id:" header;
#   the host in the "Sender:" header; else
#   the host of the poster
#
# In any event, recognize and reduce hostnames of the form
#   <id@host>
#   (id@host)
#   host(inet_addr)  [reduce to "host"]
# Make sure to remove extra words, i.e., "address (Name)"
# A hostname must have valid syntax and not match the current domain.

sub find_host {
  local ($message_host, $junk, $this, $result);

  $postmaster = "";
  $complain_to = "";
  $result = "";

  # NNTP-Posting-Host: is first choice.
  $junk = &try_host(".*NNTP-Posting-Host:", "NNTP-Posting-Host:");
  $junk = "" if &home_domain($junk);
  $result = $junk if $result eq "" && $junk ne "";

  $junk = &do_path;	# Path: header is the next most reliable.
  $junk = "" if &home_domain($junk);
  $result = $junk if $result eq "" && $junk ne "";

  # This is used in do_received, so figure it out now.
  $message_host = &try_host("^Message-Id:", "Message-Id:");

  $junk = &do_auth_warning();
  $junk = "" if &home_domain($junk);
  $result = $junk if $result eq "" && $junk ne "";

  $junk = &do_received($message_host);	# ...then, try Received:
  $junk = "" if &home_domain($junk);
  $result = $junk if $result eq "" && $junk ne "";
  print "adcomplain: WARNING: cannot use \"Received:\" headers.\n"
      if $junk eq "" && $mailmode;

  # Now curry any remaining possibilities....
  $result = $message_host if $result eq "" && $message_host ne "";

  $junk = &try_host("^Sender:", "Sender:");
  $result = $junk if $result eq "" && $junk ne "";

  if ($poster ne "") {
    print "find_host: trying poster \"$poster\"...\n" if $verbose;
    $junk = &good_hostname($poster, 0);
    &register_host("Poster:", $junk) if $junk ne "";
    $junk = "" if &home_domain($junk);
    $result = $junk if $result eq "" && $junk ne "";
    }

  if ($result eq "") {
    print "\nadcomplain: WARNING: no valid host name was found\n";
    }
  else {
    print "find_host: host is '$result'\n" if $verbose;
    }
  return $result;
  }

#-----------------------------------------------
# Revise the site name based on some exceptions and known aliases.
sub revise_site {
  local ($site) = @_;
  local ($redirect, $junk);

  $redirect = "";
  ($junk = $site) =~ s/(\W)/\\$1/g;	# remove special characters
  for (keys(%rematch_aliases)) {
    if ($_ =~ /$junk/) {
      $redirect = $_;
      last;
      }
    }
  if ($redirect ne "") {
    $junk = $rematch_aliases{$redirect};
    print "revise_site: $site redirected: '$junk'\n" if $verbose;
    $redirect = &extract_header($junk);
    }
  if ($redirect ne "") {
    $site = &find_site($redirect);
    print "revise_site:  '$redirect' is '$site'\n" if $site ne "" && $verbose;
    }
  return $site;
  }

#-----------------------------------------------
# Get all of the sites feasible from the Complain-To headers
sub do_complain_to {
  local ($junk, $junk2);
  local (@list, $i, $j, $each);

  $complain_to = "";
  for ("Complain-To", "Report-Abuse-To", "X-Complaints-To"
	) { # alternatives
    $junk = &extract_header("^.*$_:");
    next if  $junk eq "";
    print "do_complain_to: $_: is $junk\n" if $verbose;
    @list = split(/[\s,]+/, $junk);
    for ($i = 0; $i < @list; $i ++) { # process list
      $junk2 = &good_hostname($list[$i], 0);
      if ($junk2 ne "") { # good hostname
	if ($i == 0 && $#list == 0) {
	  &register_host("$_:", $junk2);
	  }
	else {
	  $j = $i + 1;
	  &register_host("$_-$j:", $junk2);
	  }
	$complain_to = $junk if $complain_to eq "";
	} # good hostname
      } # process list
    } # alternatives

  if ($complain_to eq "") {
    if ($site eq "") {
      print "\nadcomplain: WARNING: no valid complain_to address was found\n";
      }
    else {
      print "do_complain_to: setting complain_to to site name.\n" if $verbose;
      $complain_to = "$site\@abuse.net";
      }
    }
  else {
    if ($site eq "") {
      @list = split(/[\s,]+/, $complain_to);
      $site = &good_hostname($list[0], 0);
      print "do_complain_to: setting site to $site.\n" if $verbose;
      }
    }

  $complain_to =~ s/,/ /g;     # turn all commas to spaces
  $complain_to =~ s/\s+/, /g;  # and then back to commas
  print "do_complain_to: complain_to is '$complain_to'\n"
      if $complain_to ne "" && $verbose;
  }

#-----------------------------------------------
# Top-level routine that reads and extracts the headers from the message
sub parse_headers {
  local ($this);

  $subject = &extract_header("^Subject:");
  if ($subject eq "") {
    print "\nadcomplain: WARNING: subject not found\n\n";
    }
  else {
    print "subject is '$subject'\n" if $verbose;
    }

  if ($mailmode) {
    # mail mode forced
    if ($ARGV[0] ne "") {
      print 
	  "\nadcomplain: ERROR choose at most one of -m and a newsgroup name\n";
      exit 1;
      }
    $newsgroup = "";
    print "This is a mail message.\n" if $verbose;
    }
  else {
    $this = &extract_header("^Path:");
    $newsgroup = &extract_header("^Newsgroups:");
    if (defined($ARGV[0]) && $ARGV[0] ne "") {
      $newsgroup = $ARGV[0];
      print "newsgroup is from command line argument.\n" if $verbose;
      }
    if ($newsgroup eq "") {
      $mailmode = 1;
      print "Inferring that this is a mail message.\n" if $verbose;
      }
    else {
      $newsgroup =~ s/,/, /g;
      if ($newsgroup =~ /^[^,]+,[^,]+$/) {
	$newsgroup =~ s/(.*), ([^,]*)$/$1 or $2/g;
	}
      else {
	$newsgroup =~ s/(.*), ([^,]*)$/$1, or $2/g;
	}
      print "newsgroup is '$newsgroup'\n" if $verbose;
      }
    }

  $poster = &find_poster;
  $host = &find_host;

  $site = &find_site($host);
  if ($site ne "") {
    print "site is '$site'\n" if $verbose;

    $site = &revise_site($site);
    $postmaster = "postmaster\@$site";
    print "postmaster is '$postmaster'\n" if $verbose;
    }

  &do_complain_to;

  $site = "INSERT_SITE_NAME_HERE" if $site eq "";
} # parse_headers

#-----------------------------------------------
# return 1 if site is in list of exceptions
sub is_exception_site {
  local($site) = @_;

  #for (@site_exceptions) { return 1 if $site =~ /$_$/; }
  #return 0;

  local ($u, $l, $i, $pat);

  return 0 if $site eq "";
  $site =~ y/A-Z/a-z/;		# canonicalize to lower case
  $l = 0;
  $u = $#site_exceptions;
  for (;;) {
    return 0 if $u < $l;
    $i = int(($l + $u) / 2);

    # End if a pattern match is successful
    $pat = $site_exceptions[$i];
    $pat =~ s/(\W)/\\$1/g;		# remove special characters
    $pat = "$pat\$";
    if ($site =~ /$pat/i) {
      print "found exception site \"$site\".\n" if $verbose;
      return 1;
      }

    if ($site lt $site_exceptions[$i]) {
      $u = $i - 1;
      }
    elsif ($site gt $site_exceptions[$i]) {
      $l = $i + 1;
      }
    else {
      die "internal error in is_exception_site(\"$site\")";
      }
    }
  }

#-----------------------------------------------
# Return "" if the given address has a valid suffix, else the invalid
# suffix.
sub is_valid_domain {
  local($host) = @_;
  local($junk);

  ($junk = $host) =~ s/.*\.([^.\s]*)\s*/$1/;	# get that tail
  # for (@valid_domains) { return "" if $junk eq $_; }
  # return $junk;

  local ($u, $l, $i, $pat);

  return "." if $junk eq "";	# empty suffix...
  $junk =~ y/A-Z/a-z/;		# canonicalize to lower case
  $l = 0;
  $u = $#valid_domains;
  for (;;) {
    return ".$junk" if $u < $l;
    $i = int(($l + $u) / 2);

    # End if a pattern match is successful
    $pat = $valid_domains[$i];
    $pat =~ y/A-Z/a-z/;		# canonicalize to lower case
    if ($junk eq $pat) {
      # print "found valid domain \"$junk\".\n" if $verbose;
      return "";
      }

    if ($junk lt $pat) {
      $u = $i - 1;
      }
    elsif ($junk gt $pat) {
      $l = $i + 1;
      }
    else {
      die "internal error in is_valid_domain(\"$host\")";
      }
    }
  }

#-----------------------------------------------
# returns 0 or else a cleaned up hostname
sub good_hostname {
  local ($host, $must_be_real) = @_;
  local ($junk, $each);

  # clean up the hostname
  $host =~ s/.*<(.*)>.*/$1/;	# remove angle brackets, if any
  $host =~ s/.*\((.*)\).*/$1/;	# remove parens, if any
  $host =~ s/\(.*\)//;		# remove parenthesized text
  $host =~ s/.*@(.*)/$1/;
  $host =~ y/A-Z/a-z/;		# lowercase

  # print "good_hostname: $host $must_be_real\n";
  # return "" if !($host =~ /^[A-Z]/i);  # hostname must begin with a letter
  return "" unless $host =~ /.*\..*/;    # hostname must have at least one dot
  return "" if &is_inet_addr($host);

  # check for illegal characters
  for (
	# from rfc822
	"\\(", "\\)", "<", ">", "@", ",", ";", ":", "\\\\", "\\\"",
	"\\[", "\\]", # ".",
	# questionable ones added by Jason
	"#", "\\^", "\\&", "\\*", "=", "\\+", "\\|", "\\?", "`", "~", "\\/") {
    if ($host =~ /$_/) {
      print "good_hostname: \"$host\" has illegal char \"$_\".\n" if $verbose;
      return "";
      }
    }

  # Look at the irregular site names first.
  return $host if &is_exception_site($host);

  $junk = &is_valid_domain($host);
  if ($junk ne "") {
    print "good_hostname: \"$host\" has invalid suffix \"$junk\"\n" if $verbose;
    return "";
    }

  if ($must_be_real && !&home_domain($host)) {
    $host = "" if !&real_host($host);
    }

  return $host;
  }

#-----------------------------------------------
# Reduce a hostname to a site; i.e.:
#   If last field of $host >= 3 characters long (e.g., "com", "edu"),
#     set $site to last two fields (e.g., "company.com" or "college.edu")
#     unless three field version found in exceptions list,
#   otherwise, set $site to last three fields (e.g., "canadian.site.ca")
#     unless two field version found in exceptions list.
sub find_site {
  local ($host) = @_;
  local ($site, $site2, $site3, $junk);

  $host =~ s/<(.*)>/$1/;		# remove angle brackets
  $host =~ s/[^@]*@(\S*)\s*/$1/;	# extract domain-name part

  ($site2 = $host) =~ s/.*\.(.*\..*)/$1/;
  ($site3 = $host) =~ s/.*\.(.*\..*\..*)/$1/;
  ($junk = $host) =~ s/.*\.(.*)/$1/;
  if (length($junk) >= 3) {
    ($junk = $site3) =~ s/(\W)/\\$1/g;	# remove special characters

    if (grep(/^$junk$/, @site_exceptions) != 0) {
      $site = $site3;
      }
    else {
      $site = $site2;
      }
    }
  else {
    ($junk = $site2) =~ s/(\W)/\\$1/g;	# remove special characters

    if (grep(/^$junk$/, @site_exceptions) != 0) {
      $site = $site2;
      }
    else {
      $site = $site3;
      }
    }
  $site =~ y/A-Z/a-z/;		# lowercase
  return $site;
  }

#-----------------------------------------------
# Add the given site to the list of sites
sub register_host {
  local ($key, $value) = @_;

  $available_hosts{$key} = &find_site($value);
  }

#-----------------------------------------------
# Analyze the list of available hosts.  Generate diagnostic if 
# headers are inconsistent.
sub check_hosts {
  local ($firstkey, $secondkey, $firstval, $secondval);
  local (@order) = ("Path:", "Received:", # "NNTP-Posting-Host:",
	# "Message-Id:", "Sender:", "NNTP-Posting-User:",
	# "Return-Path", "Reply-To:", "From:"
	);

  foreach $firstkey (@order) {
    $firstval = $available_hosts{$firstkey};
    next if !defined($firstval) || $firstval eq "";
    foreach $secondkey (@order) {
      $secondval = $available_hosts{$secondkey};
      next if !defined($secondval) || $secondval eq "";
      if ($firstval ne $secondval) {
	$header_analysis .= "${bullet}Site implied by \"$firstkey\" header ($firstval)";
	$header_analysis .= " is not consistent with site implied by";
	$header_analysis .= " \"$secondkey\" header ($secondval).\n\n";
	return;
	}
      }
    }
  }

#-----------------------------------------------
# Return 1 if the argument refers to the home domain
sub home_domain {
  local ($probe) = @_;
  local ($each);

  for (@mydomains) {
    return 1 if $probe =~ /$_$/i;
    }
  return 0;
  }

#-----------------------------------------------
# perl code by Denis N. Antonioli <antonioli@ifi.unizh.ch>
# original C code by David Laur, Rainsound, 9/96,
# posted in comp.sys.sgi.apps <3246E9D5.41C6@rainsound.com>

# original comments:
# attempt to construct a fully-qualified hostname
# not likely to work on multi-homed or multi-addressed machines
sub fqdn_name {
  local($in) = @_;
  local($result) = $in;
  local($name, $aliases, $addrtype, $length, @addrs);
  local($nd, $cd);

  # Jason's original hack:  a reverse-DNS lookup often yields
  # a better (FQDN) name.
  ($name, $aliases, $addrtype, $length, @addrs) = gethostbyname($in);
  $result = $name if defined($name) && $name ne "";

  local($nd) = ($in =~ tr/././);
  local($cd) = ($name =~ tr/././);
  if ($cd > $nd) { # authorized name is longer; take it.
    $nd = $cd;
    $result = $name;
    }
  # Pick the alias with the _most_ dots
  local($alias);
  foreach $alias (split(' ', $aliases)) {
    $cd = ($alias =~ tr/././);
    if ($cd > $nd) {
      $result = $alias;
      $nd = $cd;
      }
    }

  # If our name has two or more dots, we're done!
  return $result if $nd >= 2;

  # Need to add the domain name to the host name.
  # ADCOMPLAIN_MDOMAIN env variable?
  local($domain) = $ENV{'ADCOMPLAIN_MDOMAIN'} || '';
  local($etc);

  if ($os_kind eq "unix") {
    $etc = "/etc";
    }
  elsif ($os_kind eq "os2") {
    $etc = $ENV{"ETC"};
    }
  elsif ($os_kind eq "winnt" || $os_kind eq "win32") {
    $etc = "";	# I might be able to do better here....
    }
  else {
    die "does $os_kind have an etc directory?";
    }

  # Try /etc/resolv.conf.  This will work on a surprising number of
  # machines!
  if ($domain eq '' && $etc ne "") {
    if (open(FD, "$etc/resolv.conf")) {
      while (<FD>) {
	if (/^domain\s+(\S+)\s*$/) {
	  $domain = $1;
	  last;
	  }
	# N.B. -- it is against BIND rules to have both 'domain' and 'search'
	# directives.
	if (/^search\s+(\S+)\s*$/) {
	  $domain = $1;
	  last;
	  }
	}
      close(FD);
      }
    }

  # Try /etc/defaultdomain.  This is much more questionable; this returns the
  # name of the NIS domain, which may or may not have anything to do with the
  # mail domain.  On the other hand, if we get this far, we're pretty stuck,
  # so we have nothing to lose....
  if ($domain eq '' && $etc ne "") {
    if (open(FD, "$etc/defaultdomain")) {
      $domain = <FD>;
      close(FD);
      }
    }
  $domain =~ s/^\s*(\S+)\s*/$1/g unless $domain eq '';
  $result .= '.' . $domain unless $domain eq '';
  return $result;
  }

#-----------------------------------------------
# Initialize the constants and lists.  Also sets $os_kind, because it's
# needed quite early.
sub get_constants_and_environment {
  local ($dollar_control_O);

  # control characters don't survive shar very well!
  $dollar_control_O = "\$dollar_control_O = \$\017;";
  eval $dollar_control_O;	# will be empty in perl4

  # Start by figuring out which OS we're on.  Fortunately, we don't
  # need to distinguish different Unixes (that would be the problem if
  # we used $^O, which is only in perl5 anyway).
  if (defined($ENV{"OS"}) && $ENV{"OS"} eq "Windows_NT") {
    # Always set by the operating system
    $os_kind = "winnt";
    }
  elsif (defined($ENV{"OS2_SHELL"}) && $ENV{"OS2_SHELL"} ne "") {
    # Always set by the operating system
    $os_kind = "os2";
    }
  elsif (defined($dollar_control_O) && $dollar_control_O eq "MSWin32") {
    # Windows 95.  Same perl as above, but %OS% will not be set.
    $os_kind = "win32";
    }
  elsif (-x "/bin/uname") { # SVr4
    $os_kind = "unix";
    }
  elsif (-x "/usr/bin/uname") { # BSD
    $os_kind = "unix";
    }
  else {
    die "what is your os?";
    }

  # Fix AF_INET and SOCK_STREAM
  if ($os_kind eq "unix") {
    # an apology here: the Unix perl4's at gemstone.com were installed _many_
    # years ago by someone who _really_ didn't know what he was doing.
    # The host-specific libraries just aren't, and reinstalling
    # _all_ of the perls for _all_ of the architectures is not worth my time.

    # So, what we'll do is make sure the values are properly set for my
    # Greatest Hits list of architectures.

    $junk = -x "/bin/uname" ? `/bin/uname -rs` : `/usr/bin/uname -rs`;

    if ($junk =~ /^SunOS 5.*/		# Solaris
       || $junk =~ /^IRIX*/i		# IRIX
	) {
      # solaris and irix just _had_ to be different
      eval 'sub AF_INET { return 2; }';
      eval 'sub SOCK_STREAM { return 2; }';
      }
    elsif ($junk =~ /^HP-UX.*/i  	# HP-UX
	|| $junk =~ /^AIX.*/i 		# AIX
	|| $junk =~ /^SunOS 4.*/i 	# SunOS
        || $junk =~ /^OSF1.*/i 		# OSF1
	|| $junk =~ /^Linux*/i          # Linux
	) {
      # the rest of the world, it seems
      eval 'sub AF_INET { return 2; }';
      eval 'sub SOCK_STREAM { return 1; }';
      }
    else {
      # Sorry, can't guess.  Your perl better be properly installed.
      require "sys/socket.ph";
      }
    }
  elsif ($os_kind eq "winnt" || $os_kind eq "win32" || $os_kind eq "os2") {
    # from winsock.h
    eval 'sub AF_INET { return 2; }';
    eval 'sub SOCK_STREAM { return 1; }';
    }
  else {
    die "cannot set AF_INET and SOCK_STREAM properly";
    }

  @monthNames = (
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep",
	"Oct", "Nov", "Dec");
  @wdayNames = ("Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat");

  # Now that I've discovered IEEE 1003.1, I just want to cry.  But here's
  # a modest attempt at time zones.  Complain away if you think it's wrong....
  %timezones = (
    "-1200", "IDLW, M",		# International Date Line West
    "-1100", "NT, SST, L",	# Nome, Samoa
    "-1000", "HST, AHST, CAT, HAST, K", # Hawaii Standard
    "-0900", "YST, HDT, AKADT, HADT, I", # Yukon Standard
    "-0800", "AKDT, PST, YDT, H", # Pacific Standard, Yukon Daylight
    "-0700", "MST, PDT, G",	# Mountain Standard, Pacific Daylight
    "-0600", "CST, MDT, F", 	# Central Standard, Mountain Daylight
    "-0500", "EST, CDT, AST, E",# Eastern Standard, Central Daylight,
				# Brazil/Acre
    "-0400", "AST, EDT, WST, ADT, CST, D",
				# Atlantic Standard, Eastern Daylight,
				# Brazil Western, Brazil/Acre, Chile Standard
    "-0330", "NST, CDT",	# Newfoundland Standard, Chile Daylight
    "-0300", "ADT, EST, WDT, C",# Atlantic Daylight, Brazil
    "-0230", "NDT",		# Newfoundland Daylight
    "-0200", "EDT, FST, AT, B",	# Brazil Eastern Daylight, Brazil/deNoronha,
				# Azores
    "-0100", "FDT, WAT, A",	# Brazil/deNoronha, West Africa
    "0000", "GMT, UTC, WET, UT, Z", # Greenwich Mean,
				# Universal Time, Western Europe
    "0100", "BST, WET DST, MET, CET, FWT, MEWT, SWT, N", 
				# Western Daylight, Middle Europe
    "0200", "EET, MET DST, METDST, MEST, MESZ, SST, FST, O, SAT, CEST, IST",
				# Middle Europe Daylight
				# Swedish Summer Time, French Summer Time,
				# South African Time, Israel
    "0300", "IDT, EET DST, MSK, BT, P", # Israel Daylight, Turkey, Moscow
    "0330", "IT",		# Iran?
    "0400", "MSD, ZP4, Q",
    "0500", "ZP5, R",		# USSR Zone 4
    "0530", "IST",		# India Standard Time
    "0600", "ZP6, S",		# USSR Zone 5
    "0630", "NST",		# North Sumatra?
    "0700", "WAST, T",		# Western Austrailia
    "0730", "JT",		# Java?
    "0800", "CST, SST, SGT, HKT, WST, CCT, WADT, U, MYT", # China & Taiwan,
				# Singapore Standard, Hong Kong, West Australia,
				# Maylasia
    "0900", "KST, CDT, JST, V",	# Korea, China, Japan
    "0930", "CST, CAST",	# North Australia, South Australia
    "1000", "KDT, EAST, EST, GST, W",
				# Korea,  Eastern Australia, Guam
    "1100", "EDT, EADT, X",
    "1200", "NZST, IDLE, NZT, Y",# New Zealand, International Date Line East
    "1300", "NZDT",		# New Zealand Daylight Time?
    );

  # Valid country domains
  @valid_domains = (
      # Countries
      "uk",  # Non-standard but frequently used.
      "su",  # Non-standard but frequently used.
      "af", "al", "dz", "as", "ad", "ao", "ai", "aq", "ag", "ar", "am", "aw",
      "au", "at", "az", "bs", "bh", "bd", "bb", "by", "be", "bz", "bj", "bm",
      "bt", "bo", "ba", "bw", "bv", "br", "io", "bn", "bg", "bf", "bi", "kh",
      "cm", "ca", "cv", "ky", "cf", "td", "cl", "cn", "cx", "cc", "co", "km",
      "cg", "ck", "cr", "ci", "hr", "cu", "cy", "cz", "dk", "dj", "dm", "do",
      "tp", "ec", "eg", "sv", "gq", "er", "ee", "et", "fk", "fo", "fj", "fi",
      "fr", "fx", "gf", "pf", "tf", "ga", "gm", "ge", "de", "gh", "gi", "gr",
      "gl", "gd", "gp", "gu", "gt", "gn", "gw", "gy", "ht", "hm", "hn", "hk",
      "hu", "is", "in", "id", "ir", "iq", "ie", "il", "it", "jm", "jp", "jo",
      "kz", "ke", "ki", "kp", "kr", "kw", "kg", "la", "lv", "lb", "ls", "lr",
      "ly", "li", "lt", "lu", "mo", "mk", "mg", "mw", "my", "mv", "ml", "mt",
      "mh", "mq", "mr", "mu", "yt", "mx", "fm", "md", "mc", "mn", "ms", "ma",
      "mz", "mm", "na", "nr", "np", "nl", "an", "nc", "nz", "ni", "ne", "ng",
      "nu", "nf", "mp", "no", "om", "pk", "pw", "pa", "pg", "py", "pe", "ph",
      "pn", "pl", "pt", "pr", "qa", "re", "ro", "ru", "rw", "kn", "lc", "vc",
      "ws", "sm", "st", "sa", "sn", "sc", "sl", "sg", "sk", "si", "sb", "so",
      "za", "gs", "es", "lk", "sh", "pm", "sd", "sr", "sj", "sz", "se", "ch",
      "sy", "tw", "tj", "tz", "th", "tg", "tk", "to", "tt", "tn", "tr", "tm",
      "tc", "tv", "ug", "ua", "ae", "gb", "us", "um", "uy", "uz", "vu", "va",
      "ve", "vn", "vg", "vi", "wf", "eh", "ye", "yu", "zr", "zm", "zw",

      # Organization Domains
      "com", "edu", "gov", "int", "mil", "net", "org",

      # Following at http://www.edns.net/registries.htm
      # unsanctioned but currently supported by a number of nameservers
      "biz", "corp", "earth", "exp", "k12", "ltd", "lnx", "med", "nic", "npo", 
      "per", "usa", "web", "xxx",

      # Following at http://www.iahc.org/press-final.html, dated 4 Feb 97.
      # Also want to hear more before accepting these....
      # "arts", "firm", "info", "nom", "rec", "store", "web",
      );
  @valid_domains = sort @valid_domains;  # needed for bsearch to work

  # mnemonics for $recipient
  $complain_to_only = 1;
  $poster_only = 2;
  $poster_and_complain_to = 3;

  # This is a simple way to match the first letter of an SMTP id with the
  # expected hour of the day....
#  %smtp_chars = ( "A", "00", "B", "01", "C", "02", "D", "03", "E", "04",
#		  "F", "05", "G", "06", "H", "07", "I", "08", "J", "09",
#		  "K", "10", "L", "11", "M", "12", "N", "13", "O", "14",
#		  "P", "15", "Q", "16", "R", "17", "S", "18", "T", "19",
#		  "U", "20", "V", "21", "W", "22", "X", "23");

  # Exceptions List
  # Please mail additions and corrections to billmc@agora.rdrop.com.

  # DESCRIPTION
  # These are site names that don't follow the convention of two fields for
  # names with three-letter suffixes (e.g., company.com) or three fields for
  # names with two-letter suffixes (e.g., canadian.site.ca).

  @site_exceptions = (
	'ab.umd.edu',
	'acc.ca',
	'ada.se',
	'adenet.es',
	'ai.mit.edu',
	'alameda-coe.k12.ca.us',
	'algonet.se',
	'austria.eu.net',
	'aware.nl',
	'banat-crisana.ro',
	'bcm.tmc.edu',
	'bctel.ca',
	'berlingske.dk',
	'bluewin.ch',
	'bme.hu',
	'bureau1.utcc.utoronto.ca',
	'bureau2.utcc.utoronto.ca',
	'bureau3.utcc.utoronto.ca',
	'bureau4.utcc.utoronto.ca',
	'bureau5.utcc.utoronto.ca',
	'bureau6.utcc.utoronto.ca',
	'bureau7.utcc.utoronto.ca',
	'campus.mci.net',
	'canit.se',
	'casema.nl',
	'casinos.at',
	'c.4biz.net',
	'.cc',
	'ceta.es',
	'chalmers.se',
	'cisnews.cisnet.com',
	'coast.to',
	'columbus.rr.com',
	'cs.umass.edu',
	'ct.rr.com',
	'cybernet.za',
	'datanet.hu',
	'demon.co.uk',
	'dial-access.att.net',
	'dial.pipex.com',
	'digibel.be',
	'direct.ca',
	'dis.dk',
	'easynet.de',
	'econnect.ca',
	'edcoe.k12.ca.us',
	'egate.ca',
	'elektrobit.fi',
	'elp.rr.com',
	'epfl.ch',
	'escape.ca',
	'eunet.be',
	'eunet.ch',
	'eunet.fi',
	'expedia.msn.com',
	'extrabit.fi',
	'fastlane.ca',
	'flashmail.cc',
	'forfree.at',
	'fun.ee',
	'futurnet.es',
	'germany.eu.net',
	'get2net.dk',
	'glo.be',
	'glas.apc.org',
	'glue.umd.edu',
	'gmx.de',
	'gnu.ai.mit.edu',
	'greenhill.pvt.k12.tx.us',
	'gregg.world-access.com',
	'grolier.fr',
	'gu.se',
	'hawaii.rr.com',
	'hl-technik.de',
	'hol.fr',
	'home.se',
	'hungarnet.hu',
	'hvu.nl',
	'iaf.nl',
	'ibernet.es',
	'ican.ca',
	'iif.hu',
	'indigo.ie',
	'inext.ro',
	'innovplace.saskatoon.sk.ca',
	'iol.ie',
	'ireq.ca',
	'isky.ca',
	'istar.ca',
	'iteso.mx',
	'ivm.de',
	'lakeheadu.ca',
	'loki.silkspin.com',
	'lserv.utcc.utoronto.ca',
	'mail.icongrp.com',
	'mail.memorystorage.com',
	'maine.rr.com',
	'med.wayne.edu',
	'mediatech.de',
	'messer.de',
	'metronet.de',
	'midsouth.rr.com',
	'mpg.de',
	'msfc.nasa.gov',
	'nacamar.de',
	'nemo.fi',
	'netcom.ca',
	'netinc.ca',
	'netway.at',
	'neuhaus.de',
	'nordbayern.de',
	'norgesnett.no',
	'nstn.ca',
	'.nu',
	'nycap.rr.com',
	'nym.alias.net',
	'odn.de',
	'on.the.net',
	'online.no',
	'oup.es',
	'oupcan.mail.net',
	'oupchile.cl',
	'oxford.pl',
	'oxford.satlink.net',
	'oxford.spin.com.mx',
	'pangea.ca',
	'pi.se',
	'pine.nl',
	'ping.at',
	'planet.eon.net',
	'pluto.spearhead.net',
	'pop.tc',
	'psi.ca',
	'postag.de',
	'pubnet.sk',
	'rba.ch',
	'redestb.es',
	'rmcs.cranfield.ac.uk',
	'ruhr-uni-bochum.de',
	'san.rr.com',
	'schwaben.de',
	'seicom.de',
	'siemens.de',
	'siemensnixdorf.de',
	'sigov.si',
	'silsoe.cranfield.ac.uk',
	'sky.fr',
	'smart.is',
	'sn.no',
	'snerpa.is',
	'sni.de',
	'sol.no',
	'spectranet.ca',
	'sprint.ca',
	'sprintcanada.ca',
	'.st',
	'stack.nl',
	'sun-denshi.co.jp',
	'sunrise.it',
	'sympatico.ca',
	'sztaki.hu',
	'tampabay.rr.com',
	'tao.ca',
	'tele2.dk',
	'telegate.nl',
	'telegate.se',
	'teleglobe.ca',
	'telepac.pt',
	'terminal.cz',
	'terranova.fi',
	'tfi.be',
	'tfi.de',
	'tin.it',
	'.tj',
	'.to',
	'tsqware.fr',
	'twcny.rr.com',
	'ualberta.ca',
	'uk.uu.net',
	'uni2.dk',
	'uni-c.dk',
	'unibe.ch',
	'unicall.be',
	'unik.no',
	'uunet.ca',
	'usask.ca',
	'uvic.ca',
	'varney.idbsu.edu',
	'videotron.ca',
	'vma.verio.net',
	'wam.umd.edu',
	'wanadoo.fr',
	'windsor.igs.net',
	'worldnet.att.net',
	'worldnet.fr',
	'worldonline.nl',
	'wwonline.ca',
	'wxs.nl',
	'xs4all.nl',
	);
    @site_exceptions = sort @site_exceptions; # needed for bsearch to work

    %rematch_aliases = (
    'nonexistent.com',  'X-NNTP-Posting-Host:',
    'nym.alias.net', 'Message-Id:',
	);

# START EXEMPT LIST
$EXEMPT_LIST = <<'END';
M```6Q3$``6`R``%N,P`!?30``9$U``&9-@`!G3<``:XX``&V80`#4&(`!&!C
M``:99``'A64`"']F``F*9P`*4F@`"K9I``O`:@`+_6L`##=L``S.;0`.*VX`
M#PMO``^<<``0YW$`$0QR`!&*<P`3F70`%*%U`!3W=@`5@W<`%FQX`!:2>0`6
MMWI`%L(`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````!```!MP```;T`!
M`6-``0(N0`$#;$`!!&%``05B0`$&;T`!!VQ``0AG0`$)=,```&5``0MN0`$,
M+D`!#65``0YL0`$/84`!$&1``1%R0`$294`!$V1``11U0`$584`!%FQ``1<N
M0`$8=$`!&3!``1ID0`$$;$`!'')``1UO0`$>=T`!'W1``01L0`$A=4`!(F1`
M`2-O0`$.80`!)&=``24Q``$@,T`!)F5``05T0`$J;T`!*VA``2PV0`$M.4`!
M+C!``2]S0`$$=$`!,7)``3)O0`$S<$`!-'-``35W0`$V=T`!-SA``3@P``$P
M.$`!.65``01N0`$\:4`!/6Q``3YN0`$_;T`!0&Q``4%L0`%"84`!0W9``3QI
M0`%%<D`!1F1``4=T0`%(94`!27A``01X0`%+>$`!3#%``4UN0`%.;D`!'&]`
M`5!C0`%1>4`!!&Q``5-I0`%4;4`!575``3UT0`%7<D`!6&$``59O0`%9+@`!
M#69``5IE``%2=$`!7"T``0HP``$;,@`!*#@``3IK``$.;0`!1&X``4IO``%/
M<T`!7CE``00Y0`%I.$`!!&Y``01O0`%L,``!:C(``5`U``%K=D`!;6-``01N
M0`%R:4`!<VE``71E0`$Q<T`!=G-``7=E0`%X9$`!>61``7IO0`%[9``!=6=`
M`7QK0`$$<D`!?V]``8!W0`&!=$`!@F5``8-N0`&$,4`!A6,``0)N0`$,+D`!
MAW1``79I0`&*<T`!BV)``8QW0`$$90`!C7E``8XQ``&&=0`!B7=``8]P0`$$
M=4`!E&1``95N0`&684`!ES!``9AR0`$$94`!FFA``9LY``&<=$`!:7-``0YE
M0`&?:4`!H'1``:%I0`&B84`!!&E``:1N0`&E;D`!IF5``:=L0`&H;$`!J6E`
M`:IM0`&K:$`!K&,``:-T0`&M:4`!=FM``;!O0`&Q;T`!LF-``;,P0`&T,$`!
MM6E``00Y0`&W.$`!N'1``;DP0`&Z,4`!NW)``0YC``&]9$`!#G-``85R0`'`
M94`!P79``<)E0`'#:4`!Q&A``<5C0`'&84`!QV1``3$M``'(84`!R7)``2%A
M0`',;4`!S71``05N0`'/:4`!T&,``<IS``'.=D`!T6=``01N0`'5:4`!UFQ`
M`==L0`'8:4`!V6)``=IU0`';<T`!,6Q``3%T0`$.94`!WVY``>!E0`'A+@`!
M`V5``=]N0`'C8P`!W&4``=UI``'>;0`!XG5``>5A0`%+;4`!ZVE``4MF0`'M
M84`![G)``>]V0`&;;$`!\6E``?)S0`'S>$`!]&%``?5H0`$R9T`!]VE``?AT
M0`$\:4`!^F]``3%E0`'\9$`!_6E``?YT0`&;<T`"`&%``@%M0`("8D`"`V5`
M`@1N``'Y<P`!^W8``?]W0`(%8D`!FVU``@IA0`(+:$`"#&Y``51O0`(.<T`"
M#V5``A!Y0`(194`!O71``A-N0`(4:$`!!&-``A9T0`(784`"&"T``@9C``(-
M90`"$FD``A5S``'Y=T`"&71``AIL0`(@:T`!(6U``B(M0`(C94`")&=``B5A
M0`(F=$`")VY``BAT0`'794`"*FM``BMR0`(L84`"+6U``BYN0`(O:4`",'-`
M`C%E0`(R<D`",W5``C1T0`(U=4`!!"X``0,T0`(W94`".'-``CII0`([=$`"
M/&X``C9R0`(]80`"*64``CYN0`$$;T`!/2T``>QG``'P;0`!]G4``B%V``)`
M>D`"0V%``9]E0`)*;D`"2V]``7]O0`)-;$`"3G1``D]S0`)0<D`"46%``=5M
M0`)3<T`"5')``E5U0`)6;T`"5VA``EAR0`)994`"6FD``E)T0`);:$`!#F-`
M`EYE0`)?=$`"8&4``F%I0`&?-$`"-W1``F1S0`)E;T`"9G-``:1R``)A<P`!
M#G5``F@P0`$$,$`";#!``FTQ``$$,D`";FU``01I0`)Q<D`"<F%``G-C0`)T
M;4`!,65``G9T0`)W<T`">'E``GES0`)Z=4`">W!``GQM0`)];T`"?F-``G\M
M0`*`94`"@6-``H)S0`&);D`"A&E``H5A0`*&;4`"AV]``HAD0`*)>4`!,7-`
M`HMO0`*,9D`"C6Y``HYL0`$.94`"D&Y``2%A0`*2=T`"DVE``90M``**:0`"
MCW0``I%U``*4=D`"E6-``3%I0`*;;D`"G&]``IUH0`*><$`"GV%``44M``*@
M+@`!#7=``J%A0`*B:$`"I6Y``<EE0`*G:4`"J')``JEF0`*J+4`"JW-``JQY
M0`*M84`"KF$``G5I``*#;``"EF\``3QP``*F=T`"KW)``3QU0`*V<T`"MV]`
M`KAP0`*Y>$`"NFA``2%G0`*\:4`"O6Y``KXM``*[8P`!R6M``K]R0`+`=4`"
MPV5``L1C0`$R=4`"QF1``L=O0`+(<D`"R7!``LIG0`++;D`"S&E``LUT``+%
M>D`"SF-``79I0`+1=D`"TG)``M-E0`+4<T`"U65``2%N0`+7<T`"V&%``MEC
M0`+::4`"VW)``MPM0`%T:\```'5``M\N0`+@;T`"X6-``N(N0`+C94`"Y&]`
M`N5F0`+F+D`"YV%``NAD0`+I80`"SV,``M9E``+==@`"WGE``NII0`*V9D`"
M\&Q``O%E0`+R=4`!,6]``O1M0`+U>4`"]FY``O=G``+S;P`"^'9``0XQ0`$$
M+@`!#6,``;=L0`+\;T`!!&-``P!S0`,!,D`!!'E``P-A0`,$>D`!/&E``P9W
M0`,'94`#"&Q``PEA0`%L9T`#"V]``PQC0`,-:0`#`FP``P5P``,*<P`!MW=`
M`PYD0`%384`#%&5``Q5R0`,6=4`#%V]``QAY0`,994`#&G-``5-S0`,<=4`#
M'7!``QYL0`$%84`#("T``Q]M0`,A;D`#(F%``R1A0`*V=T`#)G1``R=F0`,H
M>D`!!'5``RIL0`,K;T`#+')``RUI``,E;P`#*71``RYC0`$.:4`#,G1``S-N
M0`,T80`#-6]``4!Z0`%!;``#-F]``SAI0`$.;4`#.VE``SQM0`,]=$`#/FY`
M`S]M0`$.;T`#06-``T)A``$@:4`#0V5``=UC0`-&>$`#1V9``01L0`-);T`#
M2FY``7]I0`-,9P`#2VQ``TTS``&\80`!OF(``=)C``'F9``"1&4``DQF``)<
M9P`"8F@``F=I``)I:@`";VP``K!M``+K;@`"^6\``OUP``,/<@`#&W,``R]T
M``,Y=0`#0'8``T1X``-(>D`#3G-``2%E0`-G=4`#:&5``8YI0`-J<0`#:79`
M`VME0`-L8T`!/&E``V]V0`-P<D`#<65``W)S0`-S9$`#=')``W5A0`-V-``"
M-V-``W=K0`-X=T`!;&]``WMT0`-\<T`#?6%``=YG0`-_94`#@&Q``X%Y0`."
M94`#`&1``X1I0`.%;``#@W9``X9T0`(6<D`#B6]``XIW0`.+9``#?F4``X=N
M0`.,8@`#;F<``01N``-Z<D`#C6]``8YD0`.4;D`#E6E``Y9V``.&=T`#EVU`
M`YAO0`.:;T`#FW)``YQT0`&@84`#GFE``6UT0`.@=4`#H6Q``Z)O0`.C<@`#
MGW-``Z1P0`%R>4`#IW5``ZAA0`(7;T`#JG)``79D0`&?;D`#K65``ZYI0`.O
M80`#K')``[!I0`&W84`#LW=``[1A0`.U:$`#MF9``[<M``.E8@`#J6,``ZMF
M``.Q;T`#N'1``[ED``.=<T`#OFE``7)G0`/!;T`#PFQ``\-N0`/$=T`!U6%`
M`I!A``/&94`#QV]``2%O0`/*94`#?V1``\QT0`/-;T`#SFU``0]I0`/09``#
MR&8``\MH``//=$`#T6)``O1Y0`/6<D`#UV%``]AM``,+=T`"!6)``\=O0`/<
M;$`#W7-``=]O0`/?8D`!#F5``^%W0`/B<T`#XW-``^1E0`/E;4`#`&]``^=R
M0`/H+0`#VF<``]YH``/@;@`#YG!``^EE``/%9P`#TFX``]EZ0`/J:4`!;&%`
M`_-L0`$\;T`#]6A``_8M0`/W:T`#^'1``MAC``/Y<T`#^FY``3%O0`/]>D`#
M_FE``_]R0`0`;T`$`6)``1!R0`0#84`$!&9``2%I0`0&:$`$!V@`!`)M``0%
M<T`$"&5`!`EA``/[=4`$#&]``C=Y0`0/-$`$$&E``05A0`02;4`$$W-`!!1U
M0`05<D`!=&5`!!=M0`08<D`!R6%`!!IV0`0;94`$'&Q`!!UB``01;@`$%F\`
M!!EU0`0><D`#06]`!"-T0`0D<T`$)6Y`!"9I0`0G;T`!'&]`!"EW0`0J<D`$
M*V%`!"QR0`'A94`$+FM`!"]A``0H90`#!FD`!"UO0`0P9\```')`!#5O0`0V
M+D`$-VE`!#AC0`0Y:T`!#G)``P!P0`0\<D`$/65`!#YT0`0_;D`$0&5`!$%I
M0`*084`$0W=``9MO0`1%80`!R6,`!$)M``1$<$`$1FM`!$=H0`$\;@`$.W-`
M`BII0`1-;$`$3W-`!%!S0`1194`$4FY`!%-C``1,:4`$5&,`!#ML``1+<T`$
M56)``79A0`1:8@`$6W=``3QE0`1<=$`$7F$``Y!E``._:0`#[VH``_1L``0-
M;P`$'W(`!#%T``0Z=0`$5WE`!%]P0`$.<D`$:F]`!&MC0`1L:$`!#VQ``T8N
M``$#-$`$;VD``MAL0`1P;4`!U6Y`!'1E0`1U9$`$=BT``GLN0`$-84`$>&Q`
M!'IL0`1[94`$?&%``9IN0`$.;T`$?W-`!(!R0`%L84`$@F)`!(,M0`2$=$`$
MA65`!(9U0`)H94`$B&=`!(ED0`2*:4`$BW)`!(QC``1^;``$@7``!(=T0`2-
M-``"-W1``WQH0`22>4`#WVQ`!)5A0`2694`$.V5`!)AR0`298T`$FF5`!)MM
M0`$\;T`$G6A`!)YN0`2?84`$H&U`!*%B``*V8P`$;FP`!')M``1W<``$?7(`
M!(YS``24=``$EW8`!)QY0`2B-$`";71`!*UC0`$A94`$KW)`!+!I0`2Q9``$
MLF]``G$M0`+6;$`!MV%`!+8T0`04<D`$N')``5-T0`2Z<T`$NVE`!+QG0`2]
M94`$OG)`!+]L0`3`84`$P64`!+ER0`3"8P`#0G1`!,-C``2U9@`$MVD``0YN
M0`3%+4`"V&4``A=N0`+7;@`$RW1`!,QF0`$\84`$T&-`!-%I0`329D`$*BU`
M!-1N0`35;T`$UFY`!-=G0`&D94`$V6U`!-IO0`3;<D`$W'!`!-UM0`$99$`$
MWRU`!.!E0`3A9T`$XFY`!.-E0`3D;$`$Y79``X9L``3F=$`$YVQ``=YA0`3J
M8T`$ZW!`!.QA0`3M;4`#(65`!.]C0`3P:4`$\7)``VAO0`3S9D`$]'-`!/5A
M0`3V;4`$]W1`!/AS0`3Y:4`$^GE``_5T0`3\<T`$_6M`!/YN0`3_+0`$V"X`
M!-YA``3H90`$[F\`!/)R``3[=4`%`&E``<EH0`4(8T`%"7)`!0IO0`4+<D`!
M!G1`!0UN0`4.94`%#V-`!1!D0`4194`%$FE`!1-F0`44:4`%%7-`!19N``4,
M<T`%%W9``A-K0`$Q8T`%&W5`!1QB``4=:4`!;&M`!1YC0`4@:$`$.&=`!2)L
M0`'[90`%)&U`!!-B0`4E80`%&&4`!1II``4A<``%(W5`!2=R0`*5=$`%+7-`
M!2YE0`4O;T`"<6-`!3%E0`4R;$`%,V5`!31T0`4U:0`%-FY``0YN0`$/:4`%
M.6Q`!3IT0`4[<T`%/&-``Z%E0`4^;D`%/VY`!4!O0`5!8T`%0F]`!4-O0`5$
M=$`%16%``G9E0`5'<D`%2&1`!4DM0`5*;D`%2RU`!4QS0`5-;$`%3FE`!4]A
M0`5080`%1G(``==T0`51:T`%4F_```!D0`56+D`%5W1`!5AE0`59;D`%6BY`
M!5ML0`5<94`%771`!5YE0`5?:4`!4&)``01U0`5B<$`%8VA`!61T0`5E;$`%
M9F%`!6=E0`5H=T`%:6Y`!6IY0`$.=$`%;&E`!6UN0`5N;P`%:W5`!6]O0`$%
M<T`%<G9``01R0`5T:4`#*BX``0TT0`(W8@`%=F,``01R0`5W94`%>7,`!75T
M0`5\+0`%<W5`!7UF``5A;0`%<'!`!7]R0`,S=$`%A'-`!'1T0`6&;@`%A7!`
M!8=E0`6(=$`!<F5`!8MS0`6,;4`%C6]`!8YT0`&58T`%D&5`!9%O0`$=;4`%
MDV1`!91E0`65=$`%EF%`!9=D0`68:4`%F6Q`!9IO0`6;84`!#G1`!9UA0`6>
M9``%GW!`!#QT0`6@8T`%HF]``95R0`6D9T`%I6]`!:9P0`6G;0`%J&Y``=]A
M``6C94`%J6,`!8ID``6/;@`%DG(``C=S``6<=$`%JVQ``9]O0`6S;T`%M'1`
M!;5L0`6V9D`#<&9`!;AO0`6Y94`%NG1`!;MA0`6\<D`%O6]`!;YP0`6_;T`$
M!G-`!<%E0`7"80`%/6(``^1C``559``%8&T`!8%N``6M;P`%MW(`!<!V0`7#
M,0`!!&-``SMR0`$<;T`%SV9`!=!W0`7190`!(6]``8YM``');D`%TW1`!=5I
M0`779$`%V&%``V]L0`7:84`%VW!`!=QL0`7=84`%WG1`!=]S0`7@80`%TF4`
M!=EY0`7A=$`!B6Y`!>5E0`7F;``!!&X``>!O0`%`8P`%YVE`!>AC``$A90`"
M7W-``01R0`$Q94`%\'9`!?%A0`7R<D`%\V-`!?1E0`%3<T`%]G1`!?=E0`(7
M=$`%^2U`!?IE0`7[9T`%_&1`!?UE0`7^9T`%_VY`!@!I0`8!=$`&`FT`!?5R
M``7X=$`&`WA``79O0`8'9T`!=F%`!@EM0`%V9$`!/&%`!@QI``8+<D`&#6%`
M!<]Z0`80:4`&$6$`!3YF``8(<``&"G0`!@YW0`82<T`"%FE`!AAM0`899$`#
M]6Y`!AMU0`8<;D`#;V%`!AYR0`8?84`&(&5`!B%L``8B;T`!;&E``ZQS0`8E
M=D`!#VE`!B=E``8F<@`&*'5`!@QA0`$Q;$`&+&Q`!BUL0`5L:4`&+VU`!C!A
M0`8Q<D`!B65`!C-M0`8T;T`"MF-`!C9D0`8W;0`&-7)`!C@N``$-,P`!!#5`
M`01T0`8[<T`&/F$`!CEO0`8_;D`"%VE``3QK0`9#;D`&1'5`!D5A0`+>:4`&
M1V1`!DAO0`&:<D`&2G)`!DMN0`$R94`&36U`!DYE``9):0`&3&]`!D\U0`$$
M=$`&4V5`!E1T0`%3<D`&5F5``7]E0`%+80`&5V4`!EAL0`99;$`#06%`!EUE
M0`9>=4`!!G-`!F!N``9A<@`"TW1``3%L0`5C8T`&97!`!F9O0`9G90`&8F@`
M!FAP0`%3:4`$ZG)`!FQH``9M:4`&"W9``:!B0`+\<D`!L'1`!G)S0`9S=4`&
M=&1`!G5N0`9V:4`&=V1`!GAL0`9Y<D`&>F$`!G!E``9Q;T`&>RT`!A-A``8:
M8@`&'6,`!B-D``8I90`&+F8`!C)H``9`:0`&0FH`!D9M``90;@`&57``!EIR
M``9?<P`&:70`!FYW0`9\<D`&?V5`!I!F0`,`;D`&DFE`!I-R0`:494`&E6(`
M!I%P0`:6+0`$;6$`!*-B``2N8P`$LV4`!,=F``3.9P`$TV@`!0%K``%T;``%
M*&T`!3!N``4W;P`%Q'``!<UR``7B<P`%ZW0`!>UU``8$>4`&EV%``T-T0`:L
M;T`&K7=``0YA0`:O;$`&L&%``]]L0`:R:4`#_G1`!K1A0`:U94`&MG)`!K=P
M0`,`94`&N61``GML0`:[;T`&O&U`!?%A0`:^8@`&LV,`!KAD``:Z9P`&O71`
M!K]A``;`94`%.V5``6UG0`;';D`&R'5`!LED0`;*<T`&RVY`!LQC``+\:P`&
MKFX`!K%T``;%=T`&S6]`!A!T0`1^<T`&U"U`!M5H0`;680`$.'1`!M=U0`*2
M;T`&VF-`!MMS0`;<:4`&W61`!MXM0`;?<$`&X&A``5-X0`&)94`&XW9`!N1A
M0`;E80`%^G``!N)T0`;F;D`!_&E`!NIS0`;K84`&[&-`!NUR0`;N;T`&[W-`
M!O!D0`;Q;D`&\FE`!O-W0`;T+4`&]71`!O9I0`;W;T`&^')`!OET0`'K;@`&
M^W5``2$M``4R;T`&_&$`!MAE``;A;``&YVX``3%T``;Z=D`&_GI``:1A0`<&
M;$`'!W!`!PAW0`<):4`!(2U`!PMG0`'M;T`'#7)`!#MA0`</<D`!165`!Q%P
M0`/^=4`'$V]`!Q1C0`<5;$`"OG)`!Q=A0`<8=$`'&2T`!Q9S0`<:;$`'&V%`
M!QUL``<.;0`'$',`!Q)T0`<><D`$*6D`!Q]O``&4<$`'(W)``V]O0`<G9D`'
M*&Y`!REE``%3;T`'*FQ`!RMC0`,A+4`'+G1`!R]C0`<P94`',7E``4%R0`<S
M94`'-'9`!S5O0`<V8T`'-V0`!PQG``<D;``'+7(`!S)S0`<X,0`!!&Q``01L
MP```;D`'0"Y`!T%E0`="=$`'0V%`!T1G0`=%=$`'1F5`!T=N0`=(+D`'23$`
M!TIS0`+5+@`!#6$``;=S0`=+;$`!FVQ`!U!E0`=1+@`!`VD``2%S0`=2;D`'
M4VE`!U9A0`=7:0`!!&U`!UAAP```8T`'6RY`!UQE0`==;$`'7F)`!U]A0`=@
M8T`'86]`!V)G0`=C8T`'9&=``0]D0`=F:4`'9W)`!VAB0`=I=T`':FE``3%S
M0`=L>4`';6Q`!VYA0`=O;D`'<&%`!W%M0`=R84`'<RX`!V5A``=K90`'=&Q`
M`9IT0`=":$`'>6=`!WII0`=[;$`'?&5`!WUD0`=^+4`'?VA`!X!C0`>!=$`'
M@FY``HQA``;.8@`&TV4`!P!F``<*:``#,FD`!SEM``<^;@`'36\`!UER``=U
M=0`'@WE`!X1N0`-&>D`'D6E`!Y)F0`7Q9D`'E&(`!Y-M``03;@`"UV\`!Y5T
M0`9984`&,W1`!YMI0`/];P`!0',`!YQW0`>=:$`'GG1`!Z%D0`$.;D`'HV%`
M!Z1L0`>E=@`!/'E``0YA0`>G;P`!;'=`!ZET``>F>4`'JG(`!Z)S0`>L>4`!
M(&Y`![!O0`>Q<T`$.&1`![-R0`>T80`'M6]``T%B0`$Q8D`'N&Y`![EE0`>Z
M;T`$NG1`![QN0`>]94`'OG9`![]N0`?`:4`'P65`!\)V0`?#:4`'Q'1`!\5C
M0`?&94`'QV9`!\AF0`$J<T`"WF8`!\IH``$$;4`'RW1``09I0`?/=D`'T"U`
M!]%N0`?2;$`!=F%`!]1S0`?590`!<F]`!]9T0`&W:4`'V6-`!]IC0`04:4`'
MW&Y`!]UI``?;;T`'WG)`!]]T0`?A8T`'XG)`!1MO0`?D=T`'Y6%``5-J0`?G
M:4`'Z&$`!]-D``?790`'XV8`!^9L0`?I84`!R65`!^]L``?P;4`"+F<``MAN
M0`&E84`'\V%``5!M0`?V94`']V1`!_AN0`?Y9T`!/&%`!_LM``?Q+@`!`VT`
M!_5O``?Z<``'_'=`!A)L0`?]9P`"V&D`"`-P0`+8<D`%^F5`"`=W0`@(;T`(
M"61`!^=O0`@+=$`(#&$`"`1K``$\<``("G5`"`UZ0`&;:4`($FU`"!-Y0`@4
M9T`(%7)`"!9T0`&%;D`(&&5`"!EM0`@:;D`(&VE`"!QA0`@=80`$KW1`"!YR
M0`@?94`((64`"!=I``$.=$`((F5`!AYS0`@F94`()V,`!3%R0`@H;$`']G-`
M""MI0`@L<T`(+6]`""YP0`$\:4`&#'5`"#$M0`$_;D`(,V]`"#1A0`2Z9T`(
M-FQ`"#=A0`@X+0`(-6,`"#ER0`+T9P`(,G-`"#IT0`@]<D`(/V$`"#!O0`A`
M;T`%,G-`"$-S0`A$94`(160`!+)R0`A&8P`(07!`"$=N0`?[84`(2VA`"$QC
M0`A->$`(3F5`"$\M0`A0<T`(471`"%)R0`A3<$`!E&]`"%5G0`A6;D`(5VE`
M"%AD0`A9;T`(6G-``\1S0`A<94`(764`"%1L``A;<D`(7BX``0TV0`$$;T`(
M8G)`"&0N``$#<$`(97)`"&9O0`AH=$`(:6-`"&IA0`AK<D`(;'``"%]T0`AM
M=4`&M6Q`"'!O0`AQ<T`(<BU`!I1E0`AT;D`(=6Q``G$M``AS,@`!#FD`"'9L
M``$$;0`(=VX``01P0`0\+0`'EF$`!ZYB``>R8P`'MF0`![MF``?):0`'S&P`
M!^IM``@.;@`((V\``T%P``@I<@`(+W,`"$EX``AN>D`(>&Y`!<)O0`B/8T`(
MD"U``>%Y0`B2;$`(DVE`")1S0`%T94`(EFE`")=S0`B884`(F71`")IA0`4Y
M;$`!,G5`")US0`B>94`(GVP`")QO``$]<D`(H'1`"*%D0`>1;D`(I6E`"*9L
M0`BG8D`(J&Q`"*EA0`BJ8T`'O6%`"*QF0`BM84`(KFY`"*]L``B1;0`(E6X`
M")MS``BD=``(JW5`"+!R0`+894`(MW1`"+AN0`BY:4`(NF-``T=U0`B\<T`(
MO6Q`"+YA0`B_90`$RVE`",!C0`C!;D`(PW)``=5E0`C%8D`(QF$`",1E0`C'
M94`"DFU`",IUP```84`(S"Y`",UM0`C.;T`(SV-`"-`N0`C194`(TG)`"--A
M0`C480`(RW=`"-5M0`C684`!W6Q`"-EW0`$];T`(VS$``01T0`C<<D`(W6M`
M`M=R0`C@84`(X6]`!7)H0`CC8T`(Y&,`"-IE``C?;0`(XG-`".5N0`7Z;T`(
MZFE`".MH``CF<T`([')``WUE``CO;D`%PV%`!1MI0`9-;T`(\V\`"/)P0`CT
M94`(]2X``0UA0`.A8D`#\V$``0YB``(3;@`(R'(`"-AS``CM=``(\'8`"/=X
M``CX>D`(^GA``MAL0`0J9D`)!64`"01H0`D&<T`)!VM``>!R0`D*84`)"VU`
M"0QA0`D-9T`!#F%`"0]P0`D0:$`!S6$``@!C0`D280`)"64`"0YI``D1;P`)
M$W5`!+IR0`@H<$`)&F1`"1ME0`D<<T`)'75`"1YO0`$\+@`"XV9`"2!I0`?\
M;T`!<G-`"21S0`DE84`))F1`"2=N0`DH84`)*65``3QR0`DK;T`!O2X``0,R
M0`)N94`)+FY`"3!U0`DQ9``)*F8`"2QM``DM<P`(O71`"3)A0`=">D`).&%`
M"3EL0`DZ<$`).RY`"3QC``D?90`)(6P`"2-R``DS=0`!O7A`"3UV0`$J84`)
M1')`"45T0`E&+4`)1V5`"4AC0`E);D`)2F-``7]I0`E,<D`)369``TEU0`E/
M=$`)4'5``KUO0`E2:$`)4V%``Z%C0`E584`)5FT`".)S``E1=``)5'9`"5=R
M0`&:84`)7'1`"5TT``(W<T`)7FU`"5]O0`EA+@`!`W!`!@IW0`EC;T`)96Q`
M"69L0`EG94`):"T`"5AD``EB<``(574``FAV``-K>4`):65`!AAL0`EP9D`)
M<6A`"7)D``E.90`):G-`"7-I0`0O=$`)=VY`"7AA``E+90`)=&]`"7ER0`*^
M8@`)?6Q``B-I0`9,+@`!`VU`"8!B0`5V94`)@W)`"81U0`F%;``)?FX`"8%T
M0`F&80`(L60``<EF``B[:0`(^VP`"15N``04;P`)/G(`"7IS``%!=0`)AW=`
M`;=U0`$A9D`#\WE`"99A0`F7+@`!#7=`"9AE0`F9;0`)E71`"9MC0`:U=4`)
MGF1`"9]O0`F@<D`)H6<``01P0`FB<T`%,FD`":5Y0`&?<@`#QW-`":9H0`=L
M+@`!`V4`":AG0`FJ=$`!L&E`":YT0`&<;$`)L&%`";%W0`FR94`)LV=`";1C
M``FO<D`)M6%`!)UN0`FX+4`)N6%`";II0`(7<D`)O&=`";UN0`F^:4`)ORT`
M";MR``F\=$`)P&X`":MO``FV=$`)P6%``91E0`G':$`)R'1`"<EN0`G*;T`!
M4V4``=YR0`G,:$`#E'-`"<]E0`G0;$`'YW!`"=)I``G1<T`)TVQ`"=1A``G+
M;``)S7)`"=9C0`5N<T`)VG-`"=MK0`A18T`)W6]`"=YT0`G?<T`)X#!``0XP
M0`GB,$`)XV]``=5L0`GE80`)YG-`!MQI``GG;D`!,6Y``=]R0`&4;T`)[&-`
M">UH0`GN=$`)[VQ`"?!A0`GQ94`)\C)``FYH0`GT8T`)]64`"?9H0`--+0`)
MX3(`">1D``GI90`)ZV@`"?-T0`GW;$`)^6%`"?]B0`H`80`)W&0`!6)O0`H!
M9P`!"71``P!G0`$Q84`*!V9`"@AS0`H)94`*"G1`"@MA0`H,:$`*#6%``2%N
M``$,;T`$-BY`"A!N0`H294`*$V,`"@]D0`H484`#BF]`!6QB0`H8+4`*&6Q`
M"AIE``H7;T`*&RT`"APN``$#;D`"UV1`"AYR0`'\8D`*(FE`"B-E``,@=$`*
M)&Q``Y1L0`HG94`**"T`"@5D``H.;``*%6\`"B%P``G2=``*)7E`"BEL0`.)
M84`*,65`"C)F``%+:$`*,VQ``51U0`HV9D`*-V5`"CAC0`HY84`%LV5`"CMD
M0`H\=$`*/6%`"+PM0`H_9$`*0&Q`"D%R0`I";T`*0W=`"D0N0`I%9T`*1F$`
M"CYG0`I':4`)#S@`"C1A``HZ90`*2&<`"DIO``.)=4`""FQ``TUA``F<8P`)
MHV4`"<1I``G7;``*`F\`"BIR``I+<P`*47E`!0UT0`04:4`$G71`"EQY0`I=
M<$`*7FY`!1MU0`I@:$`*86=`!:9N0`IC:4`*9'1`"F5E0`IF:T`*9W)`"FAA
M0`II+0`"+VU`"FIS0`IK:4`*;6(``0YD``IB<D`*;G``"E]R0`IO=$`&XFQ`
M"G1A0`IU94`*=G=`"G=N0`IX>4`*>6A`"GIT0`I[;$`*?&%`"GUC0`G':4`*
M/7)`"H!I0`(*90`*@69`"H)H0`J#;D`"`&5`"H8M``I_9P`*A71`"H<N``$-
M<T`&U&1``:5E0`J-;4`*CBU`"E%Z0`J0:4`*D2T`"H]B``J2;4`$$V5`"I-K
M0`&5;D`%;#9``6EY0`J99$`*FF%`"IME``04;$`*G'1``4MD``DK;$`!UVT`
M"I9O``J7<``!,7(`"IAT``J==0`*GW=`"J!I0`-G;$`*J65`"JIS0`)D:T`*
MK&<`"JMN0`JM<D`!WV%`"K!M0`JQ<D`*LF5`"K-P0`JT,``*6V$`"G)E``I^
M9@`"V&D`"HAK``J+;0`#(6X``SMO``JB=0`*KGE`"K5V0`%R80`*P7-``0YX
MP```;4`*Q"Y`"L5T0`K&94`*QVY`"L@N0`K)8P`%Y71`"LIE0`K+;D`*S6Y`
M!3MO0`K/+4`*T&$`"LYS0`K184`"9&4`"M1I``&]=$`!#G1``_1R0`K894`*
MV71`"MIC``$';D`*VW9`!(EO0`K>+@`!#6]``0YG``K@;$`!!2T`!RDN0`$-
M<T`!1FQ`"N9U0`KG80`*XF(``7)G``KD<``*Z'-`"`MV0`$.94`!''1`"N]I
M0`KP;4`*\6E`"O)L0`KS;D`*]'5`"O5E0`KV;4`*]V@``01O0`KX8D`!/'5`
M"OMT0`K\;$`*_6%`"OYI0`K_<D`+`&X``>!T0`L!<T`+`FD`!+%U0`L$:0`!
M"VY``0QB0`&:94`+"2X`"P=N``L*=``%^7=``J%T0`L+<$`!_&1``8EA0`L1
M+0`+$&-`"Q)T0`0O;D`+%6D`"Q9K0`'7<D`(S"Y`"QES0`L:;D`"9&]`"QQI
M0`L==$`+'F%`"Q]I``L;;4`+('5``5!O0`LC<D`+)&=`"R5R0`LF94`+)V1`
M"RAN0`LI=$`"7V%`"RLM``L7+@`!`S0`!!!L``'[<@`+(74`"RIW``LL>D`%
M^6D`"Q-O0`LM=4`!/&Y`"S=E0`LX94`!,G)`"SIC0`L[94`+/'-`"STM0`L^
M<D`+/V5`"T!D0`M!94`$RW)`"T-I0`M$:0`+0G!`"T5M0`-!;T`+2&Q``O1P
M0`M*>4`+2W1`"TQI0`M-<D`+3FE`!'1L0`M0<D`!)7!`"U)I0`96=D`+5&E`
M"U5T0`M68T`+5V5`"UAN0`M9;D`+6F]`"UMG0`/]:4`+77-`"UYE0`M?<D`!
MFVE`"V%U0`MB<4`+8V,``;=N0`MD9``+8&E`"V5A0`&):4`+:61`"VIE0`MK
M;4`#_G)`"VUE0`MN+0`+9V8`"2QM``ML<T`+;W1`"W!X0`$.90`+=&E`"W5A
M0`,R;4`!E&%`"WEH0`+?+D`+>VU`"WQO0`M]8T`+?BY`"W]E0`N`=D`+@7-`
M!K1R``N"<T`+@V5`!Z-E0`N&90`+A'!`"X<M``M380`!2V,`"UQN``MV<``+
M>'(`"WIS0`N(9P`+3VP`"U%R0`N*94`&)F1`"Y1E0`N5=$`+EF%`"Y=M0`N8
M=\```'1`"YHN0`N;=$`+G&5`"YUN0`N>+D`+GW1`"Z!E0`NA;D`+HF,`"TEE
M``N1:``!#VD`"YEO0`NC-P`!!&$`"NYC``KY9``+!64`"P]F``LU9P`+.7$`
M`VES``M&=$`+I"X``0-C``%R9$`$%&T`!#AN0`$Q<D`'76%`"[AS``LW=$`+
MN6Q`!Z-R0`N\;P`+O7-``7)A``K"8P`*TF0`"M5E``K<;``*WVT`"NEN``NI
M;P`+LW``"[9Q``.G<P`+NG4``71W0`N^;T`!,G!`"\UK0`O.8T`+SV5`!'YC
M``GM9T`+T6I``01E0`/?8D`+U6%``V=M0`O7;4`#"V5`"]EL0`O:;$`+VV]`
M"]QH0`O=;D`+WBX``0-M0`(N94`+X')`"^)U0`OC=$`+Y&Y`"^5E0`OF=D`+
MYVY``Y1S0`OI;0`!''0`"^AU0`OJ;D`+ZV(``F1E``O8:``+WVE`"^YF0`+>
M>D`!UV%`"_1M0`OU;D`*76$`"_9I0`OW=$`+^&X``P!S0`OZ80`+T&,`"])J
M``O4;0`+UF\`"^]S``OS=4`+^V]``25Z0`P$84`,!6U`#`9A0`P';4`)+65`
M#`EL``P(<D`,"FD``;=S0`(O94`%&FQ`"95S0`P0>4`,$6M`#!)N0`P3<D`#
M1G!`#!5X0`P694`,%V5`#!AG0`P99$`,&F5`#!ML0`P<=T`,'6$`!6)O0`P>
M:$`&#7-`#"%L0`PB;T`,(RU`!:9I0`PE9T`"$V5`#"=I0`PH;D`'76]`#"HN
M0`PK;4`,+&]`#"UT0`PN<T`,+W=``3%E0`PQ;D`,,G-`##-E0`5B=T`,-6$`
M#`MC``P-90`,#VD`#!1J``$%;@`,'V\`#"1P``PF<@`,*70``7)U``PP=P`,
M-'I`##9U0`(J<$`,1&U`#$5O0`Q&8T`,1V%``2MD0`Q)8D`,2G1``P!U0`Q,
M84`,37-`#$YN0`Q/;T`,4'-`#%%M``,A>$`!3'-`#%-A0`Q59T`,5F5`#%=V
M0`Q88T`!G&E`",ID``$4;@`,6G)`#%MA0`*+<$`,7WE`#&!M``Q+<@`,4G,`
M#%EU``Q<>D`,861``WYR0`QG84`,:&)`#&DN0`QJ<D`,:V-``MAI0`QM<D`,
M;G1`#&]A``QL8T`,<&5`!E=O0`$B8D`,=&=`#'5N0`QV:4`,=VY`#'AT0`QY
M:$`,>F[```!K0`Q\+D`,?6Y`#'YI0`Q_=T`,@"U`#(%N0`R"+0`,@V5`!Z-K
M0`R$9T`*46Y`#(=I0`R(=$`,B6]`!0TR0`(W<T`,C&Q`#(UR0`R.:4`,CW-`
M`3QU0`R1;T`,DFA`#)-E0`R4<D`,E6]`#)9H0`R7+@`!#6,`#(MG``R0=T`,
MF&5`#)EB``QS9P`,>VX`#(9P``+9<P`,BG9`#)UE0`C+9T`,I&%`#*5N0`RF
M84`,IVU`#*AN0`RI;T`,JFE`#*MT0`RL84`,K6E`!3)T0`0X:$`,L&=`#+%I
M0`RR94`,LW=`#+1I0`OW90`,M71`#+9S0`N#:4`,N6U`#+IM0`R[;T`,O&-`
M#+UC``RN9P`,KW(``<ES``RW=T`,OFU``G%O0`S$8T`,Q6=`#,9D0`&P84`,
MR&Q`#,EY0`S*=$`,RW-`#,PU``Q(80`,8F0`!?!E``QQ:0`,GF\`#+]T``S'
M=4`,S71`"JEN0`S694`,UVE`#-AC0`S9<T`,VG!`!4=M0`S<+@`,VV%`#-US
M0`?V8T`,X&E`#.%A0`-,8T`!^WA`#.1O0`&4;T`,YBX``0-B``SC8P`+5&4`
M#.5L``SG;0`(XG``!#QQ``$$=P`%=GI``D-K0`$\:4`,\G)`#/-T0`ST<T`,
M]6P`#.AN0`SV<T`#`&D``7-S0`7!;T`,^BT`#/ES0`S\=$`,_6E`#/]E0`T`
M80`"_'E``01D0`T"9D`!FV9`#05A0`T&94`-!VA`#0AC0`T)<T`-"G5`#0MO
M0`T,94`!;&U`#0YO0`T/=T`-$'1`!?%S0`T284`-$RX``0-M0`T49T`-%6Y`
M#1=C0`1,:4`-&3$``00R``T1:0`-&&Y`#1IT0`T;94`-'V%``?IL0`TA=4`-
M(G-`#2-N0`TD;T`-)6-`#29N0`TG84`-*&E`#2EC``T-:0`"V&L`#2!T``TJ
M>4`"V&U`".)P0`8*=T`-,6]`#3)L0`TS;$`--&5`#35Y0`TV<D`--V5`#3AS
M``TP=$`-.71``:1A0`T\:T`"V&-`#3YI0`T_<D`-0&5`#4%P0`5R9``,WF<`
M#.)I``SW:P`-`6X`#01R``TK<P`-.G0`#3UV``U">$`-0W9`!71F0`3O84`-
M3W-``2-N0`U19P`"V&P`#5!O0`U2+@`!#7-``2%W0`U6=4`!4S,``0YB``U9
M8T`)QV%`#5II0`U=9D`"0V%`#5]C0`2O:T`!FV]`#6)R0`UC80`-86(`#61S
M0`7!=$`-96Y`#6AA0`UI:$`-:F1`!9UC``UK:4`-;&5``F1G0`UO84`-<'-`
M#7%A0`&`<$`-<V]`#71R0`UU9``-7F<`#6!M``$E<@`-;7,`#7)T0`UV<T`(
M<V4`!Q%Y0`U]<T`-?F]`#8!O``%L<D`-@71`#,1H0`V$9T`-A6E`#89N0`V'
M84`'Y'!`#8ES0`V*90`!#FP`!K1T0`V+;$`';&E`#8]B0`V084`-D2T`"H]H
M0`6=8P`-@F0`#8AL``V,<@`-DG,`#9-T0`+8=$`*/VD``05L0`$%;T`!B7)`
M#9YA``V<<``-GW=`"C-M0`$_;T`-HV-`#:1A0`VE;0`-H'-`#:9S0`,[94`#
M_61`#:II0`VK84`-K&U`#:UP0`'7<$`-KV]`#;!H0`VQ<T`-LFY`#;-L``,J
M;0`-KG)`#;1E0`VU=T`$.V%`#;EC0`;B<D`-NV%`#6)E0`$U;D`,D65`#;\N
M``$#80`%/F<`!:5M``V]<@`-OG,`#<!W0`$?>4`-P6$`#;QE0`W(=$`"0W-`
M#<LM0`W,+4`-S6Y`#<YA0`8+;D`-T&1`#=%N0`W284`-TW)`#=1N0`7V;T`-
MUFU`#=<M``W8:4`&DV=``9MA0`W;;D`-W&%`#=UM0`W>94`-WV=`#>!A0`WA
M9T`-XF(`#=5E``W9=$`-XV]``_-C0`0\:4`-Z&,`#>=M0`WI;D`-ZFE`#>QA
M0`WM=$`-[FY`#>]D``VX:``-NFX`#<EO``W/<@`-Y'5`#?!W0`BX;T`-]V(`
M`P!E``04;4`(XF=`#2%R0`W\94`-_71`#?YN0`W_<$`%VW1`#@%E0`X":T`.
M`W)`#@1A0`X%8P`,Q68``01I``X`;4`.!F9``0YE0`E$;$`.#"T`#@UX0`$$
M:4`.#G1`#A!B0`U9=$`.$F(``01F``X+;``.$7-`#A-L0`M4:4`.&&)`#AEI
M0`X:<T`.&VE`#AQS0`)Q:T`.'F5`!!1N0`X@=T`.(6E`!1QU0`<G90`#2F]`
M#B1M``AW;P`.(G``#B-S0`XE80`-1&(`#4YC``U39``-6&4`#7=I``V5:P`-
MFVP`#:=M``5B;@`-J6\`#?%P``WX<@`-^7,`#@=T``,!=0`.%'8`#AUW``X?
M>4`.)V-`!A!E0`X^<D`!WFE`#D!G0`Y!;$`.0F]`#D-O0`Y$:$`.16-`#D9S
M0`Y'+0`.2&T`!!-N0`'@>4`.26D``01P``6==$`.3&E``1UU0`Y09T`.46Q`
M#E)A0`Y3;D`.5&]`#E5I``Y6;4`*[VE``T-T0`Y9;0`./W,`#DUT``Y7=4`.
M6F1``<EO0`Y?<$`.8&E`#F%H0`YB<T`.8VY`#F1O0`YE:4`.9G!`#F=M0`YH
M84`.:6A`#FIC0`YK84`.;&M``9]R0`YN;T`.;V\`!Y5W0`YP-D`!#F-``S)E
M0`7R90`"<6E`#G5M0`&W90`*AF@`#G9O``YX<D`&MW=`!A!R0`Y];T`.?G1`
M`0]I0`+T;D`.@6$`#H!E0`Z";D`"H6]`#H5S0`Z&8D`!GV%`#HAM0`$E;T`.
MBG)`#HMT0`)>=4`.C65`#HYK0`%V84`.D'9`!?9R0`Z2:P`%;&P`#H]T``Z1
M=4`.DW=`!?IT0`9*84`.F64`#IAG0`Z:90`"DFE`#IMN0`E594`.GVA`#J!T
M0`ZA:T`.HG)`#J-A``5L90`'Y&]`#J1L0`6=;$`.J&E`#JDM``YQ-0`.<V$`
M#G1B``5V8P`.>68`#G]G``Z#:0`.AVP`#HEP``Z,<P`.E'8`#IUW``ZE>D`.
MJG=``VAD0`0X;4`%IG5`#KMI0`Z\;D`.O6Y`#KYE0`Z_;$`.P&Q`#L%E``ZZ
M:4`.PFE`!?)L0`[%94`.QF1`#L=EP```9$`.R2Y`#LIA0`[+<T`.S&E`#LTN
M0`[.;D`.SV]`#M!E``=G;0`.PW,`#LAT0`[19``#`6\`!<)R``%R=``.JW8`
M#KEW0`[28@`!=&Q`#F!R0`4R9T`%DV$`#MYC0`[?:4`"7W=`#N)A0`Z`9P`.
MY'!``LIA0`[E<@`.XW9`#N=RP```9D`.ZBY`#NMM0`[L;T`.[6-`#NYA0`[O
M=D`.\&Q`#O%A0`[R8T`.\RY`#O1U0`)?;P`.]G,`"M%V``-K>``!#GI``01U
M0`.L=$`._&-`#OUI0`[^<$`._WA`#P!L0`8V84`/`F4`#P%G0`\#94`-JG,`
M#P1T0`\&94`/!V1`#PDM``&W80`.6V(`#FUE``[6:``.W&D``G%M``[@;@`#
M,F\`#NAP``,[<P`.]70`#O=U``\*>4`#P2U``3%C0`\984`#265`#QML0`\<
M+4`/'6Y``P!I0`\?<T`/(&%`#R%C0`\B<T`/(VE`#R1K``\><T`/)7-``BIE
M0`\H=D`/*6Y`#RII0`\K94`/+')`#RUO0`\N:$`/+W-`#S!F0`\Q84`&&F]`
M#S-I0`\T84`&&')`#S9T0`\W9$`/.&Q`#SEE0`\Z:4`/.V9`#SQL0`\];4`%
MG6]`#S]H0`]`84`/06Q`#T)U0`/]9T`/1&0`#T5M0`6=;D`!H"U``@]D0`])
M;D`/2F%`#TLM``],+@`!#6-`!3%G``6E;0`"+G-`"+UZ0`$.:4`/4V]`!J\M
M``]08@`/5&Y`#U5E0`]6;D`/661`")YA0`];-$`/7&D`#UIY0`]=;4`$:F%`
M#V!E0`+7<D`/8G1`#V-S0`]D94`/96A`#V9E``]-;``/7G(`#V%T``]G=D`)
M5VY``B1Y0`6F=$`/;FE`#V]N0`]P=4`/<71`#W)R0`]S;T`/=&]`"94M``]V
M94`"D&4`#VUP``]U=$`/=W-`!UUY0`]\<T`/?6E`#WXN0`]_<T`/@&E`#X%R
M0`^":0`/@VU`##9P0`?\;T`+O7=`#X=R``^&=$`/B&E``9IA0`^+94`/C&A`
M#XUT0`^.<D`/CV5`#Y!Y0`$\90`/DFY``VAS0`^3;$`/E7-``EYE0`^7<D`/
MF&9`#YEY0`^:+0`/&F$`#R9F``\R:``/-6D`#SYK``]#;``/1FT`#TAN``]H
M<``/>7,`#X1U``^)=@`/D7<`#Y9X0`^;94`#"V-`#ZMO0`^L+4`/K6-`#ZYI
M0`^O9D`/L&E`#[%U0`1J94`/LVU``>%Y0`R484`/MFQ`#[=E``%+<$`/N'-`
M#[ET0`1O;$`!I&]`#[UN0`&P84`/OW!`#\!M0`_!;T`/PF,`#\-D0`2R;4`/
MQ&=`#\9I0`_'9$`/R'9``O4M0`_*>D`/RV5`#\QY0`Y"80`/R6P`#\UT0`_.
M94`$YW1`#])A0`_3;D`/U&]`#]5I0`_6<T`/UV%`!6QW0`_9:$`/VF,`#[)G
M``^T;``/M6T`#[MN``^\<``/OG(`#\]S``_8=$`/VV-``MYF``_E:4`!!'!`
M"=!O0`7R;4`/Z7E`#^IN0`_K9$`"WRY`#^UY0`_N=$`/[VE`#_!C0`_Q<D`/
M\F5`#_-B0`_T>4`/]6-`#_9E0`988@`+U7-`#_AL0`_Y84`/^VY`#_QO0`_]
M+@`/]W-`#_YZ0`$^84`0`6=`$`)A0!`#;4`0!"U`$`5E0!`&=$`0!V(`!79I
M0!`(90`/Z&X`#^QR``__=$`0"3-``01E0!`/<P`0$'I``9MA0!`1;$`)@VQ`
M$!1A0!`5;4`0%G-`$!<W0`$$,T`0&7)`$!IY0`9F<T`0''-`$!UU0!`><$`0
M'VM`$"!C`!`890`0&VY`$"$M0`Q(:D`)S&5`$"9V0!`G;T`0*&P`$"EN``+7
M<T`-LG1`$"IE0!`M9D`'"V]`$"]R0!`P<$`0,7)`$#)O0!`S9D`0-&X`$"YY
M0!`U84`$!G)`$#@T``(W8P`0.6<`#D%O0`%`94`0.G)`$#YU0!`_<T`00&%`
M$$%A`!`V94`00F9``C=C0!!%;0`$.'=``MAO0`%+;D`$.VE`$$IL`!!+;T`"
M9&Y`$$QK0`'7;D`03VE`$%!H0!!1=$`04BU`$%-E0!!4=D`056E`$%9T0!!7
M:4`06&=`!DYA0!!::$`!?W1`$%QE0!!=94`!=&=`$%]I0`*280`08&\`$&%R
M0`/H<$`(TBT`$%MC``J';@`07G``$&)U0!!E<D`09F5`$&MB`!!)<``&U7(`
M$$YS`!!9=0`"%W=`$&QG0`7"=$`"G&5`$'1L0!!U:$`0=G1`$'=A0!!X;D`0
M>6]`$'II0!![<T`0?&E`$'UG0`8L+4`0?V1`$(!I0!"!84`0@G=`"]%S0!"$
M=$`0A7)`$(9O0!"'<$`0B'-`$(EN0!"*;T`0BVE`$(QT0!"-<$`0CFE`$(]R
M0!"08T`!4VY`$))E0!"39$`0E&,`$)%I`!"5>4`!,6,`$'YP`!"#<T`0EG!`
M`^EE0!"<;0`0G7I``79E0`GT;$`,&&%`$*%N0!"B;T`0HVE`$*1S0!"E<T`0
MIBX``0-D0`2R80`!16]``1YS`!"H=T`0JG1`$*QE`!"G:4`0KF-``_YT0`&?
M8T`0LG5`$+-D0!"T:0`"DF]`$+5R0!"V+@`!`W-``01N0!"Y;T`0NVE`$+QP
M`!"X=$`0O64`!-EO0!"^<T`!^V)`$,)E0!##+@`!#6(`$*!F`!"O9P`0L6T`
M$,!W0!#$90`0F6D`$)YO0!#%8D`("'(``71Y0!#.-4`!4VQ``S-M0`F5<T`0
MTV5`$-1K0`%3;T`0UF]`$-=L`!#8<$`._WE`$-ES0!#;94`!*G!`$-TU`!#1
M8@`0TG(`$-5S`!#<=$`0WFP``TUN0`+7<D`0Y&$`#]QD``_F90`0"V<``EYH
M`!`3:0`0(FH`$"5L`!!#;0`01FX`$$=O`!!M<``0<W(`$,MS`!#/=0`0WW=`
M$.9C``4Q=$`$F"X``0UI0`$$;4`%,G5`$/MT0!#\;D`0_6Q`")9A0!#_:4`1
M`'1`$0%N0!$"94`1`W-`$01S0!$%94`1!G1`$0=N0!$(80`0_FE`$0DM`!#W
M8P`"V&X`$/ET``-!=4`1"FA``D-P0!$1=T`1$F]`$1-B`!$494`!,6Y`$15U
M0`'5<D`1&&1`$1ED0!$::4`1&RU`!.]E0!$==$`1'FE`$1]C``(3:0`1%W``
M$1QT0!$@94`">VU`$25I0!$F+0`"+W1`$2=L0!$H:4`"`7A`"N]I0`%!9T`1
M+6$`$2QI0!$N+4`*JF5`$3%V0!$R;T`1,V%``8YS0!$U>4`+V7)``9]E`!$X
M>D`!!'1`$3ER0!$[;``1-V]`$3QC0`2>;D`1/VE`$4`M0!%!;$`10F%`$4-U
M0!%$9$`116E`$49N0`04<D`12'5`$4EI0`2`;D`12V$`$2IG`!$K;``1+VT`
M$31N`!$V<``1/7,`$4=T`!%*=0`13'9`"SEK0`=L;T`15VY`"%%O0!%98D`1
M6F5``B]B`!%;8T`17&1`"JEJ0!%?=T`!UVY`$6%I0!%B;$`!<F1`$61T0`89
M94`19BT`$65E``I;9D`19VM`$6AU0`$/9$`)S&E`$6UO0!%N8D`#RG)`$7!E
M0!%Q>4`!4WA`$7-B`!%C8P`1:V<`$6QM`!%O<@`)QW8`$7)X0!%T=4`"%GA`
M`5-E0!%]<T`1?FY`$7]A0!&`:4`1@7-`$8)S0!&#9$`(S&5`$84N0!&&:T`1
MAVE`$8AA`!$A8@`$%&4`$4UH`!%8:0`176L``6QN`!%@;P`1=70`$7QU`!&$
M=D`1B7E`#NQA0!&5=T`1EFE`$9<N0!&8-T`$.#E`$9IN0!&;:4`1G&D``XER
M0!&=;4`!/G1`$:!E0`RP;D`1HFQ`$:-E0!&D=$`)+&4`$:9V0`%38@`1GFP`
M$:%T`!&E=D`1IRX``0-U0`)H8T`1K65`$:]V0!&P;D`1L7!``3%I0!&S=$`1
MM&Q`$;5O0!&V;T`1MV5`"4QH0!&Y8T`1NF5`$;MB0`%3<D`1O7(`$;QU0!&^
M>D`"V'5`$<%C0`%R80`1LF@`$;AI``S8;P`1OW(`$<)S0!'#84`!3&Q``91E
M0!'+:``1S&X`##)T0!&T+0`1S6@`$<QT0!&T94`1T&Y`$=-I0!'49T`1U6Y`
M$=9R0`<+;T`1V&4`$==F0!'9:$`1VF-`$=QM`!'*<D`1W7=`$,1T0!'@94`1
MX7)`$>)P``DK<T`%+F5`$>1M0!'F;D`#,G!``MAL0!'I94`1ZFA`$>MF0!'L
M84`!'&1`$>YI``:3=$`%^79`$?!E0`8L<T`1\VX`$?1R0`$&94`1]6%``QQT
M0!'X;D`1^6%`$?IS0`.@;D`1_&%`$?UI0!'\=D`1_WA`$@!A0`7P=$`2`F<`
M#D%S0!(#<D`"%V%`$@9E0!('+0`2!'-`$@AL0`^]:4`2"V0`!LIF`!'[;``'
M]FT`$?YX`!(!>0`2"7I`$@QA`!'>8P`1XV4`$>=I`!'H;``1[6X`$>]R`!'R
M=@`1]WA`$@UR0`&D;T`2'61`$AYE0!(?<D`+AF)`!6QR0`O5;T`2(V9`$B1D
M0!(E;T`2)F]`$B=W0!(H;``2(G)`$BEO0`5E;$`2+&%``MAL0!(N<$`2+V=`
M$C!N0!(Q:4`2,G!`$C-L0`2=;$`2-65`$C9T0!(W94`2.&U`$CEP`!(T=T`2
M.F$`$B%E`!(J:0`2+6]`$CLM0`GN<D`"]'-`$D)N0!)#<$`#\W-`$D5D0`%+
M<$`21V]`$DAD0!))84`22G)`$DME0!),:P`21G9`$DUC0`==<4`24&5``T-L
M0!)2+@`247!`$E-C0`0[=4`25F)`$E<N0!)8=$`2665`$EIL0!);9T`27&Y`
M!#AO0!)><D`27V$`$D%E``=F9P`21&P`$DYM`!)4;@`277(`$F!S``(`=$`1
MO&5`!1ME0!)J<$`2:VY`$FQI0!)M;0`%^G1``6US0`@Q;D`2<6E`$G)P0!)S
M<$`"G'1`$G5A`!)O:0`2='5`$G9E0`5U<T`2>G1`$GMR0!)\;T`!=&-`$GYK
M0!)_;D`2@&%`$H%R0!*";T`#`&1`$H1O0!*%;4`2AG-`$H=D0!*(;D`2B6%`
M$HIK0!*+8T`2C&)`!%MG0!*.;D`2CVE`$I!K0!*1:4`+@W9`$I,M0!*48D`2
ME65`$I9W0!*7+D`2F'!`$IEA`!)]9@`2@VD`$HUO`!*2=$`2FF5`$+)K0!*@
M;$`"D&5`$J)O0`',<$`2I&U`$J5C`!*C:4`2IG1`$J=L0`9984`2JF1`!.=I
M0!*L9$`2K61`!^9G0`%3;T`2L&Q`$K%O0!*R;D`2LVA`$K1C0!*U94`2MG1`
M$K<M0!*X94`2N6-`$KIH0!*.8T`2O&%`$KUE0!*^9$`"V&Y`$L!I0!+!8@`2
MOW=`$L)H0!+#;@`2KW(`$KMT0!+%8P`2H68`$JEN`!*K<@`2KG5`$L9K0`'A
M<D`2SGE`!#AS0!+084`2T65`$M)A0`>C94`2U&A`$M5K`!+3<D`2UF9``4MO
M0`,I<T`2VBU`$MMY0!+<=$`2W6Q`$MYA0!+?+0`2V6E`$N!I0`.A;D`2XV=`
M$N1O0!+E8T`2YF5`$N=R0!+H:$`2Z6,`$NID0`/C:$`!UW-`$NUI0!+N;$`2
M[V)`$O!U0!+Q<$`2\F5`$O-C0!+T80`2UV,`$N%E`!+K;D`2]6%`!PQR0`I1
M94`2^V,`$OID0!+\=4`!,FD``?IN0!+_<T`3`'1`$P)R0!,#;D`*6VD`$P5Y
M0`$$80`2SV4`$O9F``GN:0`2_6\`$P1R0!,&,4`"_&=`$PYA0`)R;$`3$&-`
M$Q$M0!,2<D`3$W5`$Q1O0!,5>4`3%BU`$Q=E0!,88T`!I&E`$QIR0!,;94`3
M'&U`$QUA0!,>+4`3'W-`$R!P0!,A9P`3#VL`$QEM`!,B=$`%,6C```!T0!,G
M;$`3*&%`$REC``%_;$`!/&E`$RMA0`@P+@`!`W!`!#QT0!,O<D`3,6%`$S)C
M`!,N;4`3,W``$RUS`!,T=T`%=FM`$S9C0!,Y8D`$ZW-`$SMS0!,\94`.RVY`
M$SYI0!,_;$`30'!`$T%A0`(`=T`30RU`$T1D0!-%94`31G)`$T=U0!-(=$`3
M26-`$TIE`!,]:0`30G5`$TMS0`%!9D`*[V0`$T]F0!-080`3(V4`$RIO`!,Z
M<@`33',``G)U0!-1<P`+@W1`"?1I0!-9;0`36W=`##5D0`'7+0`!02Y``0US
M0!-?90`37G-`$V%E0!-B8T`39&5``G%R0!-F84`39VA`$VAS0!-I;D`3:F%`
M$VMT0!-L84`37G)`$VYT0!-O94`3<'-`$W%I0!-R=$`-=&5`$W1R`!-S<T`3
M=6Q`#F!W0!-X+@`!#6]`$WEM0`)V;T`3?&%`!E=B`!-Z8P`3?6P`$$MN``PR
M<$`3?G)`$W]A0`'A90`3A')`$X5E``$.=D`%]F$``0YB`!-<8P`396P`$VUN
M`!-V<``3AG)`$XAF0`$E;D`3D6E`$Y)S0!.3<T`3E&,`![EE`!*B:4`3E2T`
M`OPR`!&980`1J6(``=]C`!'$9``%8F4`$A1F`!(@:``2/6D`$F%K`!)N;``2
M=VT`$IMO`!+)<``3"'0`$U-U`!.*=T`3EBX``0UN0`+7,4`3JWE``@IC0!.N
M:T`%]G-`"TME`!.Q>$`!!'(``RIS`!.P>$`3LFM`!(AC0`J'9T`*J6]`$[EL
M0!.Z;T`3NRT`$[AN0!.\:$`3O6]`!DID0!/`=$`3P7A`$\)E0!/#<D`/'V]`
M$\5N`!/$<$`3QBU`$\=H0`'[=T`3RFA`$\MN`!/)=$`3S"T`$C`N0`$#<D`%
MG71`$]%E`!//;@`!A',`$])Y0`=M+4`!2V5`$]=T0`E5<T`3V6Y`$]IO0!/;
M:4`3W'1`$]UA`!/>:4`!UFP`$]AT0!/?<$`3X6A`#)1T0!/D84`/)&$`$[=C
M`!._90`3S6P`$]-M`!/C;@`3Y7A`$^9Y0!$=94`3[FY`$^]A``02;T`3\&U`
M$_%D0`'5;$`3]&-`#III0!/V;$`3]W!`$_AU0!/Y<D`"-W5`$_MT0`O9<$`#
MRGE`$_YE0!/_;@`4`'-``2%I`!/];T`4`6Y``8EA0!0%;4`4!FQ`%`=I0!0(
M84`4"68`!;AN0`$/=$`$[V-`%`UE0!0.9D`4#W)`%!!I0`,`9$`4$G5`%!-T
M0!04<T`4%6]`%!9T0!07;T`4&'1`!;IS0!0:90`4$6@`%!EO0!0;94`&<2T`
M$_-A`!+`8@`3]60`$_IG`!/\:``4`VT`%`IN``<2;P`4"W``%!QW0!0?=$`$
MF&Y`%"ME`!0@=4`4+&%`">YR0!0O<D`4,&E`!BQR0!0R94`4,W1`"EUE0!0U
M:T`4-FY``5-I0!0X9D`4.6$`%#%B`!0T8P`4-V8`%#IM0`D/<T`*P6D`!#AN
M0`%T94`!<G1`%$-N0`2P;0`41&Y`%$5O0!1&8T`42&P`!6)T0!1)<T`-V'E`
M%$QA0!1-8D`%\6U`%$]U0!10+@`!#6Y`%%%E0!1294`45')`%%5F0!16;$`4
M5S%``FPM`!19-P`!!&P`!VQS0`-S94`*#VA`%%YD0!1?:4`48')`%&%C0`%M
M9``43FL`#:IL`!18;0`"V6X`!6QP`!1:<@`48G1`%&,N``$#<T`"V')`%&QE
M0!1N;T`#9V]`%'!B0!1Q8T`4<FE`%'-F0!1T<T`2I60`%&]F`!1U;D`4=F9`
M"2QE0!1Z<D`"T75`%'QO0!1]<T`4?F5`%']R0!2`+4`4@6Y`%()O0!2#:4`4
MA&Q`%(5A``%+8P`&N&Q`%(9H``HS=T`'YV5`%(IA`!1W90`4>VD`%(=U0!2,
M+4`&>&9`%)%I0`&]84`4DW!`%)1D0!2594`4EG1`%)=S0!2880`&)VE`%)EC
M0`'?94`4G')`%)UI0!2>9$`4GRT`$ZTS`!.O80`3M&0``=]E`!/G:``4+6D`
M%#MK`!1`;0`406X`%$IO`!1D<@`4C7,`%))W`!2:>$`4H#5``0XY0!2P<T`4
ML6Q`#KIR0!2S;T`4M'=`%+5E0`0X=$`4MV%`%+AM0!2Y;$`,-FD`%+MX0`$$
M84`4O&<``?!M0!2^84`4OVD`%+IR0!3!=$`4PG)`#2%A0!3%:T`4QF%`%,=P
M0`QS;T`4R7)`%,IP0!3++4`4S&1`%,UE0!3.;4`4SVE`%-!A0!31;$`4TF]`
M#V!I0`$&8T`3+F,`%-1D`!35<T`4UF,`%--I0!37:T`$/6Y`%-QI0!3=;$`4
MWF5`$P5A0`/";4`4X6(`!79G`!3@;$`4XFA``I-C0!3F<D`4YV5`%.AM0!3I
M+0`4ZBX``0TQ``+>=$`%^7)``G9I0!3O=4`'N"X``0UA`!3K9@`4\&P`$$MP
M0!3Q8@`4LF,`%+9L`!3$;0`4R&X`%-IP`!3?<@`4XW-`%/)A0`3+;$`4_VQ`
M%0!A0!4!:``5`FE``MAC0`$/;``5`VY`%05O``%+<T`(<F$``<QI0`5M;D`!
M=FE`%0QL`!4-;T`.7W-`%0YA0!4094`"C')`%1)U0!43=$`5%&<`%1%I`!1C
M;D`5%6]`!3%R0!499$`5&F5`%1MT0!,;;T`5'7)`%1YA0`*X94`5(&Q`%2%A
M0!"2=$`5(V,`"SMX0!4D94`5)6(`%1QE`!4?9@`1^VP`#,EP`!4B<P`5)W1`
M!EEO0!4H90`5+WA``00R0`+\94`5,F=`%3-A0!4T;$`5-75`#1)P0!4W;4`5
M.&]`%3EC0!4Z;4`5.V]`%3QE0`8,<$`5/FU`%3]A0!5`=$`506$`$L!C``31
M<T`50F]`!]9D0!5&;$`51V5`%4AL0!"Y<D`52FE`%4MV0`'7<D`5365`%4YH
M0`9M<D`,D65`%5%V0!52:4`54VY`%51G0`>=<D`55G5`!?!E0!58>4`566D`
M%5=O0!5:+0`50RX`%4EC``;:9P`53',`%4]T`!50=0`5579`%5ML0!5=84`5
M975`%69C`!4]:0`%)'1`%6=Q0`-I;D`5:V]`%6QI0!5M8P`+5'-``01N0!5O
M:4`5<6U`%7)A0!5S9``5,&4``01G``&;;``5-G```01R`!5H<P`5;G1`%71X
M0`PV84`5?6%`#"=R`!5^>4`5?VQ`#OUA`!4&8@`5"&,`%0IE`!46:0`5=6\`
M%8!P``)Q<P`%8W4`%8)W0`)Q<T`!('E`%8UL0!6.=T`#\S(`%9!A0`K[=$`5
MD64`#)1P0`0490`3&F@`##9I0`$$=$`"`VP`%8]N`!63<@`5E',`%99T0!69
M=D`!,6]`#B1S0!6@:$`5H71`%:)L0!6C;T`!66]`#RAA`!,>9@`5I6@`%:9P
M``?G=D`2DV5`!?%R`!6L<T`"*G-``9MI0!6O90`&MG5`%;!A`!6M;P`"<7)`
M%;%M0`&;84`5MF5`%;=R0!6X<T`$[W(`%;IX0`$.9$`!=FE`%;UU0!6^80`$
M$G5`!D5R0`+\94`5PG1`%<-S0!7$80`5Q6]``_5C0!'_94`5R&\`!$5R0!7)
M<D`&)VU`!PMB0!7-=4`5SG-`%<]E0!709P`!;'1`%=%N0`&;=T`5U&9``\%I
M0!768T`5UW)``RIE0!79;D`5VF$`%=5E`!78:4`5VW)`$%]O0!7?90`5S&D`
M%=)P`!7<=$`5X'1``9IC0!7E94`5YF1`%>=A0!7H<D`5Z2T`%:=C`!6S9``5
MN64`%;MG`!6_:@`5P&T`%<9P`!7*<P`5X70`%>IW0`T0<D`(I2U`!Y%L0!7W
M;4`'O&5`%?EM0!7Z=$`5^W-`%?QA`!7];D`!!&]`%?XN``$-8P`6`'=``1]T
M0!8!80`5I&(`%>MI`!7V;``5^'-`%@1E0`00<D`6"F%`%@ME0!8,<D`6#65`
M$3AT0!8/=4`6$'!`%A%M0!82;T`6$V-`%A1E0!85;$`6%F%`%A=S0!8894`6
M&6Q`%AIE`!8.;T`6&VE``?QD0!8>=4`6'W1`%B!S0!8A9$`6(G)`%B-A0!8D
M=$`$/7-`%B9E`!8G;T`-#V,`%B5W0!8H,D`5D'9``1QL0!8M8D`6+F5`%B]R
M0!8P:4`6,6A`%C)D`!8J;``6+'-`%C-A0`+\94`6-W-`%CAN0!8Y84`6.BX`
M`0UD`!8[=$`(N'1`!UUE0!8_;D`60&0`$%]L`!8T;@`6/',`$NUZ0!9!=$`#
M*F9`%D=Y0!9(;4`-O6]`$>YL`!9+;4`$$W5`!D)A0!9.=T`%\6]`%E!A0`)?
M80`/8&5`%E)S0`Y@94`656E`%E9R0!9794`66"T``M9L`!9/<``647(`%E-S
M0!999$`66F0`%DIK`!9,;$`67W)`%F!E0`2==$`69&E`%F5J0`&:8P`!'&0`
M!#AE``)V;4`-#F$`%9IB`!6?8P`#,F0``6QE`!8%:``6'&D`%D)J``&W;@`6
M26\`%F-R`!9F<P`69W=`%FAI0`%_=$`6>6%`%GIT0!9[<T`6?&%`%!-R0!9^
M94`6?VU`%H!E0!:!<@`6@G9``01M``,A<T`2"&Y``\=R0!:'=4`6B&]`%HER
M0`?\;T`6BW1`%HPM`!:%:@`6BG-`%HUX0!:.+0`6?64``8EO``4Q<``!,70`
M%H-X0!:1;T`-4F-`%IAK0!:9:T`"$V-`%IME0!:<9T`!!F5`%IYL0!:?<T`6
MH'1`%J%I0!:B<$`6HV]`"EMH``R3:4`&DW5``51R0!:H=$`6J2T`%J9I``:3
M;0`$$VX`";AS0!:J9T`6I65`%K!G`!:E<@`6JW9`%K%U0!:R:$`2A#(`%IIB
M`!:=90`6I&\`%K5U0!:V:4`$T&Q`%KPT0!:]=$`6OG-`%K]G0`%M90`6P&\`
*!3ER``+\>4`6P0``
END
# END EXEMPT LIST
  $EXEMPT_LIST = unpack("u", $EXEMPT_LIST);
  } # get_constants_and_environment

#-----------------------------------------------
# j-random initialization, including reading environment variables
# and initializing result data.
sub initialize {
  local ($pat, $aliases, $addrtype, $length, @addrs);

  # Determine the OS, set up networking constants, establish constant
  # lists, such as legal mail domain suffixes.
  &get_constants_and_environment();

  $adcomplain_name = '$RCSfile: adcomplain.pl,v $';	# filled in by RCS
  $pat = 'RCSfile:';				# have to fool rcs
  $adcomplain_name =~ s/\$$pat (.*),v \$/$1/;
  $adcomplain_revision = '$Revision: 3.52 $';	# filled in by RCS
  $pat = 'Revision:';				# have to fool rcs
  $adcomplain_revision =~ s/\$$pat (.*) \$/$1/;

  # Determine the editor
  $editor = $ENV{"VISUAL"};
  $editor = $ENV{"EDITOR"} if !defined($editor) || $editor eq "";
  if (!defined($editor) || $editor eq "") {
    if ($os_kind eq "unix") {
      $editor = "vi";
      }
    elsif ($os_kind eq "winnt" || $os_kind eq "win32") {
      $editor = "notepad";
      }
    elsif ($os_kind eq "os2") {
      $editor = "e";
      }
    else {
      die "don't know default editor for os $os_kind";
      }
    }

  # Determine the pager
  $pager = $ENV{"PAGER"};
  $pager = "" if !defined($pager);
  if ($pager eq "" && $os_kind eq "unix") {
    # better default on Unix
    $pager = "more";
    }
  $pager = "" if $pager eq "builtin";

  # Get $local_hostname
  if ($os_kind eq "unix") {
    local($hname) = "/bin/hostname";			# most unixes
    $hname = "/usr/bsd/hostname" if ! -x $hname;	# irix :-(
    die "Cannot find your hostname command" if ! -x $hname;

    chop($local_hostname = `$hname`);
    die "$hname failure" if $local_hostname eq "";
    }
  elsif ($os_kind eq "winnt") {
    $local_hostname = $ENV{"COMPUTERNAME"};

    die "%COMPUTERNAME% is not set, what's wrong with your installation?"
	if $local_hostname eq "";
    }
  elsif ($os_kind eq "win32") {
    # I wish there was a better way of doing this....
    $local_hostname = $ENV{"COMPUTERNAME"};
    
    die "Please set the COMPUTERNAME environment variable"
	if $local_hostname eq "";
    }
  elsif ($os_kind eq "os2") {
    # I wish there was a better way of doing this....
    $local_hostname = $ENV{"HOSTNAME"};
    
    die "Please set the HOSTNAME environment variable"
	if $local_hostname eq "";
    }
  else {
    die "cannot determine local hostname on this os ($os_kind)";
    }

  # Try to make it fully qualified
  $local_hostname = &fqdn_name($local_hostname);
  if (&good_hostname($local_hostname, 0) eq "") {
    print "WARNING: unable to determine mail address of this computer.\n";
    print "         (\"$local_hostname\" doesn't look right.)\n";
    print "         Be sure to edit before sending.  Set ADCOMPLAIN_MDOMAIN\n";
    print "         in your environment to permanently rectify this.\n";
    }

  # Don't pull the hostname off of the domain.  Some mail domains require
  # even the computername in order to do proper delivery.
  if ($fqdnWithoutHost && $local_hostname =~ /^[^\.]+\.(.+)$/) {
    $local_hostname = $1;
    }


  $mailhost = $ENV{"ADCOMPLAIN_MAILHOST"}
	if defined($ENV{"ADCOMPLAIN_MAILHOST"});
  if (!defined($mailhost) || $mailhost eq "") {
    if ($os_kind eq "unix") {
      # Unix: assume sendmail is running locally
      $mailhost = $local_hostname;
      }
    else {
      die "configuration error: MAILHOST not set";
      }
    }

  $SIG{'PIPE'} = 'IGNORE';
  $header_analysis = "";
  $bullet = "*  ";	# used in formatting header_analysis
  $bullet2 = "   ";	# for hanging indent

  # Make stderr and stdout unbuffered.
  select(STDERR); $| = 1;
  select(STDOUT); $| = 1;

  if ($os_kind eq "unix") {
    $tempname="/tmp/adcomplain-$$";
    $CONSOLE = "/dev/tty";
    }
  elsif ($os_kind eq "winnt" || $os_kind eq "win32" || $os_kind eq "os2") {
    # Two weird conventions for a temp directory on Windows NT
    local($tmp) = $ENV{"TMP"};
    $tmp = $ENV{"TEMP"} if !defined($tmp) || $tmp eq "";
    $tempname="$tmp\\adcomplain-$$";	# ?? is there a better way?
    $tempname =~ s#/#\\#g;	# in case Unix-isms crept in

    $CONSOLE = "con:";
    }
  else {
    die "need tempname and console for this os ($os_kind)";
    }

  $hosts_file = $ENV{"ADCOMPLAIN_HOSTS"};
  $hosts_file = "" if !defined($hosts_file) || $hosts_file eq ""
	|| ! -f $hosts_file;

  %hosts_to_ips = ();
  %ips_to_hosts = ();

  $from = $ENV{"ADCOMPLAIN_FROM"};
  if (!defined($from) || $from eq "") {
    # username
    if ($os_kind eq "unix") { # safe to call getpwuid
      local($USER,@garbage) = getpwuid($<);
      @garbage = @garbage;	# shut up perl -w
      $from = $USER;
      $from = $ENV{"USER"} if $from eq "";			# desperation
      $from = $ENV{"LOGNAME"} if $from eq "";			# ...then panic
      die "I can't figure out who you are!" if $from eq "";	# ...then death
      }
    elsif ($os_kind eq "winnt") {
      $from = $ENV{"USERNAME"};
      die "%USERNAME% is not set, what's wrong with your installation?"
	  if $from eq "";
      }
    elsif ($os_kind eq "win32" || $os_kind eq "os2") {
      # This might be buried in the registry somewhere....
      $from = $ENV{"USERNAME"};
      die "Please set %ADCOMPLAIN_FROM% or %USERNAME% in your autoexec"
	  if $from eq "";
      }
    else {
      die "how do I get your username on this os? ($os_kind)";
      }
    $from = "$from\@$local_hostname";
    }
  if (&good_hostname($from, 0) eq "") {
    print "WARNING: your return address \"$from\" is not valid.\n";
    print "         Be sure to edit before sending.  Set ADCOMPLAIN_FROM\n";
    print "         (or ADCOMPLAIN_MDOMAIN) in your environment to\n";
    print "         permanently rectify this.\n";
    }

  # This is a dictionary of the hostnames we've found.
  %available_hosts = ();

} # initialize

#-----------------------------------------------
# Create standard text for a response.
sub insert_standard_text {
  local ($response);

  $response = "[generated by $adcomplain_name $adcomplain_revision]\n\n";

  if ($header_analysis ne "") {
    $response .= "I am complaining about ";
    if ($chainmode) {
      $response .= "an illegal chain letter ";
      }
    else {
      $response .= "a commercial advertisement ";
      }
    if ($mailmode) {
      $response .= "sent to my e-mail address.";
      }
    else {
      $response .= "posted to $newsgroup.";
      }
    $response .= "\n";
    $response .= <<'END';
This message apparently passed through your site.  It may even have originated
with you.  However, there are inconsistencies in the RFC-822 mail 
headers.  It is possible that a mail program was buggy or misconfigured, but it
is much more likely that the sender has intentionally obscured the message's
true origin.  It follows that the sender is aware of the dishonest and
illegal nature of this message.

The problems I see with these headers are:

END
    $response .= "$header_analysis";
    } # $header_analysis
  else { # legit headers
    $response .= "Please ";
    if ($recipient != $complain_to_only) {
      $response .= "don't ";
      }
    else {
      $response .= "advise this user not to ";
      }
    if ($mailmode) {
      $response .= "mail ";
      }
    else {
      $response .= "post ";
      }
    $response .= "these ";
    if ($chainmode) {
      $response .= "illegal chain letters ";
      }
    else {
      $response .= "unsolicited commercial advertisements ";
      }
    $response .= "to ";
    if ($mailmode) {
      $response .= "people's e-mail addresses.  ";
      }
    else {
      $response .= "$newsgroup.  ";
      }
    $response .= "My site must pay to receive and store them.\n";
    } # legit headers
  if (!$chainmode && !$mailmode && !$quiet) {
    $ans = &get_answer("Is \"$subject\" relevant to $newsgroup (y/n)?",
	"n");
    if ($ans eq "n") {
      $response .= "\nIn addition, \"$subject\"\n";
      $response .= "has nothing to do with $newsgroup.\n";
      }
    }
  if (!$noinclude || $recipient ne $poster_only) {
    $response .= "\nI forward the entire text of the message, with ";
    $response .= "headers, to the end of this message.\n";
    }
  $response .= "\n";

  if ($chainmode) {
    $response .= <<'END';
This sort of chain letter is a pyramid scheme, which in turn is a form of fraud.
It violates a U.S. federal statute, 18 U.S.C. sec. 1343.

If at any point the U.S. Mail is used, this is also a violation of 18 U.S.C
sec. 1302, the Postal Lottery Statute.  See

     http://www.usps.gov/websites/depart/inspect/chainlet.htm
     http://www.usps.gov/websites/depart/inspect/usc18/mailfr.htm

This sort of chain letter is outlawed in Canada under the Section 206(1)(e) of
the Criminal Code and under Section 55 of the Competition Act.

END
    }
  else { # advertising
    if ($mailmode) {
      $response .= <<'END';
These advertisements are illegal in many countries; please consult an attorney
for your applicable statutes.

END
      }
    else { # Usenet
      if ($recipient != $complain_to_only) {
	$response .= <<'END';
The biz.marketplace and ads newsgroup hierarchies are the correct places for
commercial ads. (Ads may be inappropriate in some biz newsgroups, so please
check first.)

For more information about Usenet etiquette, see news.announce.newusers.

END
	}
      } # Usenet
    } # advertising

  if (!$mailmode && $recipient != $complain_to_only) { # cancel your post
    $response .= <<'END';
Most text-based newsreaders will allow you to cancel a posting by finding your
article and pressing 'C' or 'D'.

(Netscape users can select 'Cancel Message' in the 'Edit' menu.)
If you need help, contact
END
    if ($postmaster ne "") {
      $response .= "your postmaster at\n\n";
      $response .= "     <$postmaster>.\n";
      }
    else {
      $response .= "postmaster.\n";
      }
    $response .= "\n";
    } # cancel your post

  $response .= "Thank you for your cooperation.\n\n";

  # batchmode
  if ($batchmode && !$outmode) {
    $response .= <<'END';

P.S. -- I used the questionable "batch mode" in adcomplain to create
this message.  If this message was addressed improperly, it is because I
did not proofread the list of recipients before mailing.  Direct
your complaints to me, not to the authors of adcomplain.  Thank you.
END
    }
  # format the response
  $response = &fmt($response, 72);

  return $response;
  }

#-----------------------------------------------
# compose a message out of all of the elements so far
sub compose_message {
  local ($response);
  local($header);

  ## find out where to send message
  if ($batchmode || $quiet) {
    }
  elsif ($poster ne "" && $complain_to ne "") {
    print "Send to:\n";
    print " 1 = $complain_to\n";
    print " 2 = $poster\n";
    print " 3 = both\n";
    print " 4 = <abort>\n";
    $ans = &get_answer("Send to whom?", $recipient);
    if ($ans == 4) {
      print "aborted\n";
      exit 0;
      }
    $recipient = $ans if $ans ne "" && $ans >= 1 && $ans <= 3;
    }

  if ($complaintFile eq "") {
    $response = &insert_standard_text;
    }
  else {
    $response = "";
    if ($complaintFile ne "") {
      if (-f $complaintFile) {
	if (!open(TEXTFILE, "$complaintFile")) {
	  print "adcomplain: cannot open $complaintFile: $!\n";
	  }
	else {
	  local(@text);
	  @text = <TEXTFILE>;
	  $response = join('', @text);
	  close TEXTFILE;
	  @text = (); # garbage collection
	  }
	}
      else {
	print "adcomplain: $complaintFile: no such file.\n";
	#require "getcwd.pl";
	#$cwd = &getcwd;
	#print "adcomplain: (current directory is $cwd)\n";
	}

      # interpolate perl variables into the response string
      $response = eval "\"" . "$response" . "\"";
      }
    } # $complaintFile given

  ## include copy of posting if needed
  if (!$noinclude || $recipient ne $poster_only) {
    $response .= "\n--------forwarded message--------\n";
    for (@original) {
      $response .= $_;
      }
    $response .= "-----end of forwarded message-----\n\n";
    }

  ## append signature if needed
  $siggie = $ENV{"SIGNATURE"};
  if (!defined($siggie) || $siggie eq "") {
    $siggie = $ENV{"HOME"};
    $siggie .= "/.signature" if defined($siggie) && $siggie ne "";
    }

  if (!$omit_signature && $addsig && defined($siggie) && -f $siggie) {
    if (!open(TEXTFILE, $siggie)) {
      print "adcomplain: cannot open $siggie: $!\n";
      }
    else {
      local(@text);

      $response .= "-- \n";
      @text = <TEXTFILE>;
      # in case siggie already has the delim....
      shift(@text) if $text[0] eq "-- \n" || $text[0] eq "--\n";
      for (@text) {
	$response .= $_;
	}
      close TEXTFILE;
      @text = (); # garbage collection
      }
    }

  ## prepend message header
  $header = "";

  { # Add RFC-822 "Date:" header
    local ($Date);
    local ($sec,$min,$hour,$mday,$mon,$year,$wday,$junk) = gmtime(time);

    $junk = sprintf("%02d:%02d:%02d", $hour, $min, $sec);
    $mday = sprintf("%02d", $mday);
    $mon = $monthNames[$mon];
    $year += 1900;
    $wday = $wdayNames[$wday];

    # TO DO:  we could actually put the time zone in.  For now, use GMT
    $Date = "$wday, $mday $mon $year $junk +0000 (GMT)";
    $header .= "Date: $Date\n";
  }

  # Add "From:"
  $header .= "From: $from\n";

  # who is it to?
  if ($recipient == $complain_to_only) {
    $header .= "To: $complain_to\n";
    }
  else {
    $header .= "To: $poster\n";
    }

  # and who gets copies?
  if (!$legal_letter) {
    $cc = ", $cc" if $cc ne "";
    $cc = "$authorities$cc";
    }
  if ($recipient == $poster_and_complain_to) {
    $cc = ", $cc" if $cc ne "";
    $cc = "$complain_to$cc";
    }
  $header .= "Cc: $cc\n" if $cc ne "";

  $subject = "no subject" if $subject eq "";
  $header .= "Subject: ";
  if ($recipient == $complain_to_only) {
    $header .= "Abuse report:  ";
    }
  else {
    $header .= "Re: your ";
    }
  if ($mailmode) {
    $header .= "email ";
    }
  else {
    $header .= "Usenet article ";
    }
  $header .= "titled \"$subject\"\n";
  $header .= "\n"; # RFC822 end of headers
  $response = $header . $response;

  return $response;
} # compose_message

#-----------------------------------------------
# Ask a question and get a reply from /dev/tty, use default if no reply given:
sub get_answer {
  local($prompt, $default) = @_;
  local($ans);
  local($/) = "\n";

  print "$prompt [$default] ";
  $ans = <STDIN>;
  chop($ans);
  return $ans if ($ans ne "");
  return $default;
  }

#-----------------------------------------------
# format a string into paragraphs with given maximum line length.
sub fmt {
  local($str, $len) = @_;
  local(@lines) = split(/\n/, $str);
  # local(@paras) = split(/\n{2,}/, $str);
  local(@paras);
  local($para, $result, $left, $word);
  local($last_was_bullet, $is_bullet, $junk);

  # split into paragraphs
  @paras = ();
  $para = "";
  for (@lines) {
    if ($_ eq "\n" || $_ eq "") {
      # blank line delimits beginning of paragraph
      push(@paras, $para);
      push(@paras, "");
      $para = "";
      next;
      }
    $_ =~ s/\n/ /g;
    if ($_ =~ /^[\s]+.*/) { # so does a line beginning with white space
      push(@paras, $para) if $para ne "";
      $para = "";
      }

    # Join the line, using proper respect for punctuation
    if ($para =~ /.*\.\s*$/) { # ends in a period
      $para =~ s/(.*\.)\s*$/$1  /;
      }
    elsif ($para =~ /.*:\s*$/) { # ends in a colon
      $para =~ s/(.*:)\s*$/$1  /;
      }
    elsif ($para =~ /.*!\s*$/) { # ends in an exclamation
      $para =~ s/(.*!)\s*$/$1  /;
      }
    else {
      $para .= " " if $para ne "";
      }
    $para .= $_;
    }
  push(@paras, $para) if $para ne "";

  # Reformat the paragraphs
  $result = "";
  $last_was_bullet = 0;
  $junk = $bullet;
  $junk =~ s/(\W)/\\$1/g;		# remove special characters
  $junk = "^$junk.*";
  for (@paras) {
    $left = $len;
    $is_bullet = ($_ =~ /$junk/) ? 1 : 0;

    # special case:  indentation at beginning of paragraph is preserved
    if ($_ =~ /^\s+/) {
      ($word = $_) =~ s/^(\s+).*/$1/;
      $left -= length($word);
      $result .= $word;
      }

    # Now start adding the words
    while ($_ =~ s/^\s*(\S+\s*)//) {
      $word = $1;
      if (length($word) < $left) {
	$result .= $word;
	$left -= length($word);
	}
      else {
	if ($is_bullet) {
	  $result .= "\n$bullet2$word";
	  }
	else {
	  $result .= "\n$word";
	  }
	$left = $len - length($word);
	}
      }
  $result .= "\n" if !$last_was_bullet;
  $last_was_bullet = $is_bullet;
  }
return $result;
}

#-----------------------------------------------
# display a screen's worth; used by moref().
sub display {
  local($my_name, $start, $count, @lines) = @_;
  local($i, $max);

  if ($start + $count <= $#lines) {
    $max = $start + $count;
    }
  else {
    $max = $#lines;
    }
  for ($i = $start; $i < $max; $i ++) {
    print $lines[$i];
    }
  printf "- $my_name: %d/$#lines: ", $start + 1;
  }

#-----------------------------------------------
# a cheap internal pager
sub moref {
  local($tempname) = @_;  # the file to display
  local($my_name) = "moref";
  local ($line_count) = 0;
  local ($lines_per_screen) = 22;
  local ($kbd_cmd, $front, $back);
  local (@all_lines);

  if ($pager ne "") {
    if ($os_kind eq "unix") {
      system "$pager $tempname <$CONSOLE";
      }
    elsif ($os_kind eq "winnt" || $os_kind eq "win32" || $os_kind eq "os2") {
      system "$pager $tempname";
      }
    else {
      die "how do I run your pager on this system?";
      }
    return;
    }

  if (defined($ENV{"LINES"}) && $ENV{"LINES"} != 0) {
    # Unix termcap convention
    $lines_per_screen = $ENV{"LINES"} - 1;
    }

  open (MOREF, "<$tempname") || die "$my_name: can't open $tempname:  $!"; 
  @all_lines = <MOREF>;
  close MOREF;

  # At this point it's still possible that some of the lines may be oversize,
  # because we don't format the original mail message from the sender; we go
  # the entire message and break up any oversized lines.
  for ($i = 0; $i <= $#all_lines; $i ++) {
    next if length($all_lines[$i]) < 80;
    $front = substr($all_lines[$i], 0, 79);
    $back = substr($all_lines[$i], 79);
    $all_lines[$i] = "$front\n";
    splice(@all_lines, $i + 1, 0, $back);
    }

  $line_count = 0;
  for (;;) {
    &display($my_name, $line_count, $lines_per_screen, @all_lines);
    $kbd_cmd = <STDIN>;
    chop($kbd_cmd);
    if ($kbd_cmd eq "") {
      $line_count += $lines_per_screen;
      $line_count = $#all_lines if $line_count > $#all_lines;
      next;
      }
    elsif ($kbd_cmd =~ /^\?.*/) {
      print "? - this message\n";
      print "<enter> - next page\n";
      print "b - back a page\n";
      print "nnnn -- start with given number line\n";
      print "q - quit\n";
      print "press enter to continue... ";
      $kbd_cmd = <STDIN>;
      }
    elsif ($kbd_cmd =~ /^b.*/i) {
      $line_count -= $lines_per_screen;
      $line_count = 0 if $line_count < 0;
      next;
      }
    elsif ($kbd_cmd =~ /^q.*/i) {
      last;
      }
    elsif ($kbd_cmd =~ /^\d+$/) {
      $line_count = $kbd_cmd - 1;	# take input as one-based line number
      $line_count = $#all_lines if $line_count > $#all_lines;
      next;
      }
    else {
      print "what? (\"?<Enter>\" at the prompt for help): ";
      print "for now, press enter to continue... ";
      $kbd_cmd = <STDIN>;
      }
    }
}
